<?php
/*
  Adds "attached file" box to your text. Works similar to DokuWiki's <php file.php>
  tag but is more flexible and works with any markup.
  Supports both inline and block forms. Files are stored in a customized path and
  named according to their content's MD5 hash so if the same contents is attached
  several times only one file is stored.

    %%(attach Hello.php; php)
      echo 'Hello, world!';
    %%

  Supports parameters:
    * hide - if set, attached contents won't be shown - only a link will be created;
      Useful for binary data (e.g. retrieved by a preceding formatter, see an example
      below). If you supply a formatter this setting has no effect:
      %%(attach file.wiki, hide; wacko) - 'wacko' is explicitly given overriding 'hide'.
    * no stats - hides "Download X times" string (for block form) or info hint (for inline);
    * mime - a custom value for Content-Type header sent when user is downloading
      attachment; you can add "; charset=XXX" after it for text documents.
      Example: %%(attach post.wiki mime="text/wacko; charset=cp1251").
      If none given the browser is let to detect it by itself.
      * don't forget that you can't mix several charsets in one wiki document even
        if you're using {{Charset}} so don't put attached contents in a different
        charset than the document itself.
    * compress - name of PHP function to pass to ob_gzhandler(); is restricted to
      $AttachmentCompression setting; if no value given (used as a flag) the first handler
      listed in $AttachmentCompression, if any, is used.
      * if $DefaultAttachmentCompression is set and you want to turn compression off for
        this specific attachment pass a dash or an empty string: %%(attach compress=)

  Examples:

    %%(attach book.fb2, compress)
    %%(attach sample.json, mime=application/javascript, no stats)
    %%(attach sample.json, application/javascript)  - the same, "mime" can be omitted
                                                      if it follows the name.
    %%(data f2w-d00.rar; attach back.rar, hide)

  There's a second action included that outputs a number of downloads for a particular
  attachment in this or another document. Corresponding %%attach call doesn't have
  to be made before its download count is retrieved. File names are case-insensitive.

    {{DownloadCount attachment.txt}}                - accesses current doc's attachment
    {{DownloadCount md5=123..., format=txt}}        - accesses a custom file by its MD5/format pair

  CONFIGURATION - fields added to UWikiSettings:

    * $AttachmentPath - path when attached files are stored; if unset %%(attach)
      will show an error message (the same happens for $AttachmentScript).
    * $AttachmentScript - URL for a script that will serve attachments for downloaded.
      This file can be used as one if you copy it to a location accessible from the Internet
      and point $AttachmentPath there, making files to store in the directory of this script.
    * $AttachmentCompression - array of allowed PHP handlers for use with 'compress' argument.
      By default is array('ob_gzhandler') which permits %%(attach compress=ob_gzhandler)
      (or just 'compress' since it's a flag and ob_gzhandler is the first listed handler).
    * $DefaultAttachmentCompression - name of method; has to be listed in $AttachmentCompression
      and compress.php (see below). If this is null or false compression is disabled unless
      %%(attach compress[=XXX]) is given; otherwise it's that argument's default value.
    * $DefaultAttachmentMIME - an array of default MIME types based on extension given.
      E. g. if this is array('wiki' => 'text/plain') then %%(attach file.wiki; wacko)
      automatically gets text/plain. If %%(attahc mime=XXX) is given it overrides this.
      Default MIME for all file exts not listed there can be set as array('' => 'defualt/mime'),
      if it's unset (by default) the browser decides which MIME to use.

  Node that $AttachmentCompression is set for document settings; however, when an attachment
  is being downloaded using this script there's no way to tell the genuine value of that
  setting (request variables can be tricked). For this reason to allow something different
  than just 'ob_gzhandler' create a file named "compress.php" ("php" extension is used
  to prevent this file from being viewable from the web) in $AttachmentPath and put
  space-separated list of allowed PHP functions, e.g. "ob_gzhandler ob_iconv_handler".

    If compress.php doesn't exist it will be set to "ob_gzhandler"; if it does but is
    empty no compression methods will be allowed.
*/

class UWikiAttachments {
  static function File($path, $file) {
    return rtrim($path, '\\/')."/$file";
  }

  // returns null if there was no record in the stat file.
  static function DownloadCountOf($file, $path) {
    if ($statFile = self::File($path, 'stats.php') and is_file($statFile)) {
      $stats = file_get_contents($statFile);
      $pos = $namePos = strpos($stats, " $file\n");
      if ($pos !== false) { while (is_numeric($stats[--$pos])) { } }

      if ($pos !== false and $stats[$pos] === "\n") {
        return (int) substr($stats, $pos + 1, $namePos - $pos);
      }
    }
  }

  static function IncDownloadCountOf($file, $path, $delta = +1) {
    if ($statFile = self::File($path, 'stats.php')) {
      if (is_file($statFile)) {
        $stats = file_get_contents($statFile);
        $count = self::DownloadCountOf($file, $path);
        if (isset($count)) {
          $stats = str_replace("\n$count $file\n", "\n".($count + $delta)." $file\n", $stats);
        }
      } else {
        $stats = "\n";
      }

      if (!isset($count)) { $stats .= "$delta $file\n"; }

      return file_put_contents($statFile, $stats, LOCK_EX) > 0;
    }
  }

  static function IsAllowedCompression($method, $path) {
    if (ltrim($method, 'a..zA..Z0..9_') === '' and function_exists($method)) {
      $file = self::File($path, 'compress.php');
      $list = is_file($file) ? file_get_contents($file) : 'ob_gzhandler';
      if (strpos(" $list ", " $method ") !== false) { return $method; }
    }
  }

  static function StatsStringFor($file, UWikiSettings $settings) {
    $full = @$settings->strings['%%attach: stats'];
    $full or $full = "(downloaded %s)";
    $dl = @$settings->strings['%%attach: stats downloads'];
    $dl or $dl = "$ time, s, , s, s";

    $count = self::DownloadCountOf($file, $settings->AttachmentPath);
    $dl = self::FmtNumUsingString($dl, (int) $count);

    return sprintf($full, $dl);
  }

  static function FmtNumUsingString($fmtString, $number) {
    $rolls = substr($fmtString, -1) === '*';
    $inflections = explode( ',', rtrim($fmtString, '*') );  // $ хит, ов, , а, ов*
    foreach ($inflections as &$str) { $str = trim($str); }

    $stem = array_shift($inflections);
    return str_replace('$', $number, self::FmtNumUsing($stem, $inflections, $rolls, $number));
  }

   static function FmtNumUsing($stem, array $inflections, $numberRolls, $number) {
      $inflection = '';

      if ($number == 0) {
        $inflection = $inflections[0];
      } elseif ($number == 1) {
        $inflection = $inflections[1];
      } elseif ($number <= 4) {
        $inflection = $inflections[2];
      } elseif ($number <= 20 or !$numberRolls) {
        $inflection = $inflections[3];
      } else {  // 21 and over
        return self::FmtNumUsing( $stem, $inflections, $numberRolls, substr($number, 1) );
      }

      return $stem.$inflection;
    }
}

if (class_exists('UWikiBaseElement')) {
  class Uattach_Root extends UWikiBaseElement {
    static $defaultMIME = array(
      'rtf' => 'application/rtf', 'txt' => 'text/plain', 'wiki' => 'text/plain',
      'wacko' => 'text/plain', 'log' => 'text/plain', 'csv' => 'text/csv',
      'htm' => 'text/html', 'html' => 'text/html', 'php' => 'text/plain',
      'xml' => 'application/xhtml+xml', 'css' => 'text/css', 'xsl' => 'application/xml',
      'ini' => 'text/plain', 'inf' => 'text/plain', 'conf' => 'text/plain',
      'cnf' => 'text/plain', 'manifest' => 'text/plain', 'c' => 'text/x-c', 'cpp' => 'text/x-c',
      'h' => 'text/x-c', 'hpp' => 'text/x-c', 'pas' => 'text/x-pascal', 'rb' => 'text/plain',
      'class' => 'text/x-java-source', 'java' => 'text/x-java-source', 'inc' => 'text/plain',
      'sh' => 'application/x-sh', 'py' => 'text/plain', 'bat' => 'text/plain',
      'tar' => 'application/x-tar', 'gz' => 'application/x-gzip', 'bz2' => 'application/x-bzip2',
      'rar' => 'application/x-rar-compressed', 'zip' => 'application/zip',
      '7z' => 'application/x-7z-compressed', 'exe' => 'application/x-msdownload',
      'dll' => 'application/x-msdownload'
    );

    public $isFormatter = true;
    public $isAction = false;
    public $htmlTag = 'span';
    public $htmlClasses = array('attachment');
    public $parameterOrder = array('name');

    public $noStats;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);
      array_push($props['bool'], 'noStats');
    }

    function IsDynamic() { return !$this->noStats; }

    function Parse() {
      if ($format = $this->settings->format) {
        $this->isBlock = $format->blockExpected;
        $format->root = $this;
        if (!self::IsEmptyStr($format->current['params']['name'])) {
          $this->SetupUsing($format);
        }
      } else {
        // absense of format chain means it's called directly as a markup, e.g. TryParsing(..., 'scripter')
        $this->isBlock = true;
      }

      $this->isBlock and $this->htmlTag = 'fieldset';
      parent::Parse();
    }

      function SetupUsing(UWikiFormat $format) {
        $params = $format->current['params'];
        $hide = (!empty($params['hide']) or self::IsEmptyStr($format->raw));

        if ($hide) {
          $this->htmlClasses[] = 'hidden';
          $markup = $this->DefaultStyle();
        } else {
          $format->IsEmpty() and $format->AddFormat($this->DefaultStyle());
          $markup = $format->NextMarkup();
          $this->htmlClasses[] = $markup;
        }

        $ext = $this->pager->FormatFrom($params['name']);
        $this->htmlClasses[] = $ext;

        $urlOptions = array_intersect_key($params, array('mime', 'compress'));
        $saved = $this->Save($this->raw, $params['name'], $markup, $urlOptions);
        if ($saved) {
          list($url, $md5, $fileName) = $saved;

          $format->origDoc->AttachmentFileByName[ mb_strtolower($params['name']) ] = $fileName;

          $file = $this->children[] = $this->NewElement('Uattach_FileName');
          $format->blockExpected and $file->htmlTag = 'legend';
          $file->isBlock = $format->blockExpected;
          $file->name = $params['name'];
          $file->file = $fileName;
          $file->url = $url;
          $file->htmlAttributes['title'] = strtoupper($md5);

          empty($params['no stats']) and $file->SetupStatsFor($fileName);
        } else {
          $this->children[] = $this->NewElement('Uattach_SavingError');
        }
      }

    function Save(&$data, $name, $markup, array $options) {
      $path = rtrim($this->settings->AttachmentPath, '\\/');
      file_exists($path) or mkdir($path, 0770, true);

      if (is_dir($path)) {
        $md5 = md5($data);
        $file = "$path/$md5.$markup";
        is_file($file) or file_put_contents($file, $data);

        if (is_file($file) and $url = $this->settings->AttachmentScript) {
          $url = trim($url, '?&');
          $url .= strpos($url, '?') === false ? '?' : '&';

          if (!isset($options['mime'])) {
            $defaultMIME = &$this->settings->DefaultAttachmentMIME;
            isset($defaultMIME) or $defaultMIME = self::$defaultMIME;

            $ext = strrpos($name, '.');
            $ext = $ext === false ? '' : substr($name, $ext + 1);
            if (isset($defaultMIME[ strtolower($ext) ])) {
              $options['mime'] = $defaultMIME[$ext];
            }
          }

          $compress = &$options['compress'];
          if (!isset($compress)) {
            $compress = @$this->settings->DefaultAttachmentCompression;
            isset($compress) or $compress = 'ob_gzhandler';
          }
          if ("$compress" === '' or $compress === '-') { unset($options['compress']); }

          $options = compact('name') + array('file' => "$md5.$markup") + $options;
          foreach ($options as $name => &$value) { $value = "$name=".urlencode($value); }
          return array($url.join($options, '&'), $md5, "$md5.$markup");
        }
      }
    }
  }

  class Uattach_FileName extends UWikiBaseElement {
    public $htmlTag = 'span';
    public $htmlClasses = array('file-name');

    public $name, $file, $statFile, $url;
    public $noStats = true;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);

      array_push($props['str'], 'name', 'file', 'statFile', 'url');
      $props['bool'] = 'noStats';
    }

    function SetupStatsFor($statFile) {
      if ($this->isBlock) {
        $stats = $this->children[] = $this->NewElement('Uattach_Stats');
        $stats->statFile = $statFile;
      } else {
        $this->noStats = false;
        $this->statFile = $statFile;
      }
    }

    function SelfToHtmlWith($contents) {
      $contents = '<a href="'.self::QuoteHTML($this->url).'">'.$this->name.'</a> '.$contents;
      return parent::SelfToHtmlWith($contents);
    }

      function SelfHtmlAttributes() {
        if (!$this->isBlock and !$this->noStats) {
          $title = UWikiAttachments::StatsStringFor($this->statFile, $this->settings);
          $title[0] = mb_strtoupper($title[0]);
          return compact('title');
        }
      }
  }

    class Uattach_SavingError extends UWikiFormatErrorMessage {
      public $message = '%%attach: saving error';
      public $defaultMessage = 'Cannot save attached document - check your settings.';
    }

  class Uattach_Stats extends UWikiBaseElement {
    public $htmlTag = 'span';
    public $htmlClasses = array('stats');

    public $statFile;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);
      $props['str'][] = 'statFile';
    }

    function SelfToHtmlWith($contents) {
      $contents = UWikiAttachments::StatsStringFor($this->statFile, $this->settings);
      return parent::SelfToHtmlWith($contents);
    }
  }

  class Udownloadcount_Root extends UWikiBaseAction {
    public $htmlTag = 'span';
    public $htmlClasses = array('attachment-dl-count');
    public $parameterOrder = array('name');

    public $statFile;

    function IsDynamic($format, $params) { return true; }
    function Priority($format, $params) { return (int) UWikiMaxPriority * 0.3; }

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);
      $props['str'][] = 'statFile';
    }

    function Execute($format, $params) {
      $format->appendMe = true;

      if ($md5 = &$params['md5']) {
        $format = &$params['format'];
        if (!$format) {
          $root = $this->NewElement('Uattach_Root');
          $format = $root->DefaultStyle();
        }

        $this->statFile = "$md5.$format";
      } else {
        $this->statFile = &$format->origDoc->AttachmentFileByName[ mb_strtolower($params['name']) ];
      }
    }

    function SelfToHtmlWith($contents) {
      $count = UWikiAttachments::DownloadCountOf($this->statFile, $this->settings->AttachmentPath);
      return parent::SelfToHtmlWith((int) $count);
    }
  }
}

/*
  Request variables:
    * file - name of file to send including extension; only [A-Za-z0-9.] are allowed;
    * name (optional) - value for Content-Disposition header;
    * mime (optional) - for Content-Type;
    * compress (optional) - method to use, is restricted by compress.php;
*/

if (count(get_included_files()) < 2 or defined('ServeWikiAttachments')) {
  $path = rtrim(ServeWikiAttachments === true ? getcwd() : ServeWikiAttachments, '\\/');
  $file = $_REQUEST['file'];

  if (!$file) {
    echo 'No file name given.';
  } elseif (ltrim($file, 'A..Za..z0..9.') !== '' or strpos($file, '.') !== 32 // MD5 hashes are 32 chars long.
            or !is_file("$path/$file")) {
    echo "File '$file' doesn't exist.";
  } else {
    ignore_user_abort(false);

    UWikiAttachments::IncDownloadCountOf($file, $path);

    $name = &$_REQUEST['name'];
    "$name" === '' and $name = $file;
    header("Content-Disposition: attachment; filename=\"$name\"");

    if ($mime = &$_REQUEST['mime']) { header("Content-Type: $mime"); }

    if ($compress = &$_REQUEST['compress'] and
        UWikiAttachments::IsAllowedCompression($compress, $path)) {
      ob_start($compress);
    }

    readfile("$path/$file");
  }
}
