<?php
/*
  This action will change all (or only external) links so they will go through a specified
  URL. Useful when anonymizing external links or just for some gateway/statistic script.

  It can either be called manually or, if set in Settings will autoload for all documents.
  By default autoloading is inactive and only external URLs are proxied.
  Note: there's no default proxy URL - if it's unspecified an Ulinkproxy_NoURL exception will be thrown.

  This action accepts the following parameters:
    = proxy As the first argument; see LinkProxyURL for substituion characters ("$" and "@").
    = scope As the second argument - 'external' (default), 'local' or 'all'; see LinkProxyScope.
    = rel If passed, will add HTML "rel=" attribute; see LinkProxyRel.
    = class If passed, will change "class=" of the link; separate multiple classes with
      spaces. Classes prefixed with "!" are removed (e.g. "!external"), others are added
      unless they already exist. Default value of this is "linkproxy". See LinkProxyClasses.
    = exclude A regular expression which will be matched against incoming URLs. If doesn't start
      with '/' it gets surrounded by regexp brackets automatically (you can't specify modificators).

  If value for a parameter isn't passed it will first be taken from Settings (see CONFIGURATION)
  and if none found there either the default values will ultimately be used.

    {{LinkProxy http://an.on/?$&from=us}}
    {{LinkProxy go.php?url=$&caption=Source:+@, scope=all}}
    {{LinkProxy no external links please!, external}}        - you can even mask out external links completely
    {{LinkProxy @, rel=nofollow, class=!external nofollow}}  - or be more democratic and only add "nofollow" to them

  You can combine two {{LinkProxy}} calls (or one call and autoload) in one document:

    {{LinkProxy @, rel=nofollow, scope=external, class=!external proxied}}
    {{LinkProxy clickmap.php?$, scope=local, exclude=/linkstat\.php/}}

  Note: scopes shouldn't overlap in one document: e.g. external+local is fine but all+external
  might yield in something unexpected (or might not).

  CONFIGURATION - fields added to UWikiSettings:

    * LinkProxyAutoload - if true, LinkProxy will be called even if no {{LinkProxy}} existed
      in source document. It will accept default settings (LinkProxyURL, etc. below) in
      this case. If explicit {{LinkProxy}} was found this setting has no effect for that doc.
    * LinkProxyURL - pattern for proxying URL where "$" is replaced by original URL (URL-encoded).
      To specify raw "$" double it: go.php?url=$&caption=$$External$$ - actual caption arg
      is "$External$". Also, "@" stands for unescaped URL (and "@@" stands for "@").
    * LinkProxyScope - can be "external" (the default) - like ((http://links)), "local" or "all".
    * LinkProxyRel - allows to add HTML "rel=" attribute - for eample, "nofollow" for search robots.
    * LinkProxyClasses - list custom HTML classes to add (speace-separated). "linkobj" is
      added by default (if LinkProxyClasses is unspecified it's no more added so list it
      there manually if you need).
    * LinkProxyExclude A regular exporession; see the 'exclude' argument.
*/

UWikiDocument::$afterParsingHooks[] = array('Ulinkproxy_Root', 'Autoload');

class Ulinkproxy_Root extends UWikiBaseAction {
  public $parameterOrder = array('proxy', 'scope');

  function Execute($format, $params) {
    $format->settings->LinkProxy = true;

    $handler = array(__CLASS__, 'ChangeURL', $params);
    $format->settings->handlers->AddOrReplace('linkproxy-'.@$params['scope'],
                                              'linkOnRender', $handler);
  }

    static function ChangeURL(&$url, $linkObj, $params = array()) {
      extract($params, EXTR_SKIP);

        isset($proxy) or $proxy = $linkObj->settings->LinkProxyURL;
        if (!isset($proxy)) { throw new Ulinkproxy_NoURL(); }

        if (!isset($scope)) {
          $setting = &$linkObj->settings->LinkProxyScope;
          $scope = isset($setting) ? $setting : 'external';
        }

      if ($scope === 'external') {
        if (!$linkObj->isExternal) { return; }
      } elseif ($scope === 'local') {
        if ($linkObj->isExternal) { return; }
      }

      isset($exclude) or $exclude = &$linkObj->settings->LinkProxyExclude;
      if (((string) $exclude) !== '') {
        $exclude[0] === '/' or $exclude = '~'.str_replace('~', '\\~', $exclude).'~';
        if (!preg_match($exclude, $url)) { return; }
      }

      $proxy = self::SimpleFmt($proxy, $url, '@');
      $url = self::SimpleFmt($proxy, urlencode($url));

      if (isset($class)) {
        $classes = $class;
      } else {
        $setting = &$linkObj->settings->LinkProxyClasses;
        $classes = isset($setting) ? $setting : 'linkproxy';
      }
      foreach (explode(' ', $classes) as $class) {
        $class = trim($class);
        if ($class !== '') {
          $i = array_search(ltrim($class, '!'), $linkObj->htmlClasses);
          if ($class[0] === '!') {
            if ($i !== false) { unset($linkObj->htmlClasses[$i]); }
          } else {
            $i === false and $linkObj->htmlClasses[] = $class;
          }
        }
      }

      if (!isset($rel)) {
        $setting = &$linkObj->settings->LinkProxyRel;
        $rel = isset($setting) ? trim($setting) : '';
      }
      $rel === '' or $linkObj->htmlAttributes['rel'] = $rel;
    }

  static function Autoload($doc) {
    if (!empty($doc->settings->LinkProxyAutoload) and empty($doc->settings->LinkProxy)) {
      $doc->settings->handlers->AddOrReplace('linkproxy-', 'linkOnRender', array(__CLASS__, 'ChangeURL'));
    }
  }
}

  class Ulinkproxy_NoURL extends EUverseWiki {
    function __construct() {
      parent::__construct('{{LinkProxy}} should be either passed URL or has LinkProxyURL set in settings.');
    }
  }
