<?php

abstract class BaseDestinationInterface {
  public $path, $cl;

  abstract static function CanHandle($path);

  function __construct($path, $cl) {
    $this->path = rtrim($path, '\\/');
    $this->cl = $cl;
  }

  function CheckEnvironment() { }
  function CleanUp() { }

  abstract function Exists();
  abstract function AddFile($name, $data);

  abstract function IsDirectory($name);
  abstract function RemoveFile($name);
  abstract function RemoveRecursive($name);

  function CopyFile($src, $destBaseName) {
    return $this->AddFile($destBaseName, file_get_contents($src));
  }

  function RemoveByList($list) {
    foreach ($list as $name) {
      $this->IsDirectory($name) ? $this->RemoveRecursive($name) : $this->RemoveFile($name);
    }
  }
}

  class FileDestInterface extends BaseDestinationInterface {
    static function CanHandle($path) { return true; }

    function Exists() { return is_dir($this->path) and is_readable($this->path); }

    function AddFile($baseName, $data) {
        $path = $this->path."/$baseName";

      is_dir(dirname($path)) or mkdir(dirname($path), 0700, true);
      $bytesWritten = file_put_contents($path, $data, LOCK_EX);
        if ($bytesWritten === false) { throw new EDestInterfaceWrite($path, $baseName); }

      return $bytesWritten;
    }

    function CopyFile($src, $destBaseName) {
      $dest = $this->path."/$destBaseName";
      is_dir(dirname($dest)) or mkdir(dirname($dest), 0700, true);
      return copy($src, $dest);
    }

    function IsDirectory($name) { return is_dir($this->path."/$name"); }

    function RemoveFile($name) {
      unlink($this->path."/$name");

      if (file_exists($this->path."/$name")) {
        throw new EDestInterfaceRemove($path, $name);
      }
    }

    function RemoveRecursive($name) {
      RmDirRecursive($this->path."/$name");
    }
  }

abstract class BaseSourceInterface {
  public $path, $cl;

  abstract static function CanHandle($path);

  function __construct($path, $cl) {
    $this->path = rtrim($path, '\\/');
    $this->cl = $cl;
  }

  function CheckEnvironment() { }
  function CleanUp() { }

  abstract function Exists();
  abstract function FileExistsIn($root, $file);
  abstract function LocalPath();

  function UpdatedFiles($destInterface) { }
  function RemovedFiles($destInterface) { }
  function NewFiles($destInterface) { }
}

  class FileInterface extends BaseSourceInterface {
    static function CanHandle($path) { return true; }

    function Exists() { return is_dir($this->path) and is_readable($this->path); }

    function FileExistsIn($root, $file) {
      $root = rtrim($root, '\\/');
      $file = "$root/".ltrim($file, '\\/');
      return is_file($file) and is_readable($file);
    }

    function LocalPath() { return $this->path; }

    function UpdatedFiles($destInterface) {
      if ($destInterface instanceof FileDestInterface) {
        return $this->UpdatedFilesIn($destInterface->path, $this->path);
      }
    }

      function UpdatedFilesIn($path, $localPath, $prefix = '') {
        $result = array();

        if (is_dir($path) and is_dir($localPath)) {
          $path .= '/';
          $localPath .= '/';

          $files = scandir($path);
          foreach ($files as $file) {
            if ($file !== '.' and $file !== '..') {
              if (is_dir($path.$file)) {
                $updated = $this->UpdatedFilesIn($path.$file, $localPath.$file, $prefix."$file/");
                $result = array_merge($result, $updated);
              } elseif (ExtOf($file) === $this->cl->options['outext']) {
                $local = substr( $file, 0, -1 * strlen(ExtOf($file)) ).'.'.$this->cl->options['inext'];
                if (is_file($localPath.$local) and filemtime($path.$file) < filemtime($localPath.$local)) {
                  $result[] = $prefix.$local;
                }
              }
            }
          }
        }

        return $result;
      }

    function RemovedFiles($destInterface) {
      if ($destInterface instanceof FileDestInterface) {
        $toDelete = array();

          $srcPath = $this->path.'/';
          $existing = FindFilesRecursiveIn($destInterface->path);
          foreach ($existing as $file) {
            if (!is_file($srcPath.$file) and !is_dir(dirname($srcPath.$file))) {
              if (empty($toDelete[ dirname(dirname($file)) ])) {
                $toDelete[ dirname($file) ] = true;
              }
            } elseif (ExtOf($file) !== $this->cl->options['outext']) {
              $toDelete[$file] = true;
            } else {
              $srcFile = substr( $file, 0, -1 * strlen(ExtOf($file)) ).'.'.$this->cl->options['inext'];
              is_file($srcPath.$srcFile) or $toDelete[$file] = true;
            }
          }

        return array_keys($toDelete);
      }
    }

    function NewFiles($destInterface) {
      if ($destInterface instanceof FileDestInterface) {
        $dest = FindFilesRecursiveIn($destInterface->path);
        foreach ($dest as &$file) {
          if (ExtOf($file) === $this->cl->options['outext']) {
            $file = substr( $file, 0, -1 * strlen(ExtOf($file)) ).'.'.$this->cl->options['inext'];
          }
        }

        return array_diff(FindFilesRecursiveIn($this->path), $dest);
      }
    }
  }

  class ESvnCommand extends EExternalCommand { }

  UWikiTask::$interfaces['source'][] = 'SvnInterface';
  class SvnInterface extends BaseSourceInterface {
    public $localPath, $repoRoot;

    static function CanHandle($path) {
      return preg_match('~^(https?|svn|file)://.~i', $path) and ExtOf($path) == '';
    }

    function CheckEnvironment() {
      try {
        $this->ExecSVN('--version');
      } catch (ESvnCommand $e) {
        $error = 'Subversion console tool (svn.exe) is required but wasn\'t found (see --svn-command).';
        throw new EEnvironmentCheck( array($error => true) );
      }
    }

    function Exists() { return true; }

    function FileExistsIn($root, $file) {
      try {
        $root = rtrim($root, '\\/');
        $file = "$root/".ltrim($file, '\\/');
        $output = $this->ExecSVN('info', $file.'@HEAD');
        return in_array('Node Kind: file', $output);
      } catch (ESvnCommand $e) {
      }
    }

    function LocalPath() {
      if (!$this->localPath) {
        $this->cl->ProgressMessage('Checking out from '.$this->path);

        $localPath = MakeTempDir('wacko-svn');
          $this->ExecSVN('export', '-q', '--force', '-rhead', $this->path, $localPath);
        $this->localPath = $localPath;
      }

      return $this->localPath;
    }

    function ExecSVN() {
        $args = func_get_args();

      array_unshift($args, '--no-auth-cache');
      array_unshift($args, '--non-interactive');
      array_unshift($args, $this->cl->options['svn-command']);

      if (!IsEmptyStr( $login = $this->cl->options['svn-login'] )) {
        array_unshift($args, "--username $login");
      }
      if (!IsEmptyStr( $login = $this->cl->options['svn-password'] )) {
        array_unshift($args, "--password $password");
      }

        foreach ($args as &$arg) { $arg = escapeshellarg($arg); }

      $command = join(' ', $args);
      exec("$command 2>&1", $output, $exitCode);

        if ($exitCode !== 0) { throw new ESvnCommand($command, $exitCode); }

      return $output;
    }

    function FilesChangedInLastCommit($purpose, $onlyActions = null) {
      $files = array();
      $this->cl->ProgressMessage("Getting last commit's log message $purpose from ".$this->path);

        $lines = $this->ExecSVN('log', '-qvl1', '-rhead', $this->path);

        if (trim(array_shift($lines), '-') !== '' or
           (!empty($lines) and (!array_shift($lines) or array_shift($lines) !== 'Changed paths:'))) {
          throw new OtherTaskError('Wrong output of `svn log`.');
        }

        while (($line = array_shift($lines)) and trim($line, '-') !== '') {
            $file = trim($line);
            $action = strtoupper( $file[0] );

          if (substr($file, -1) !== '/' and
              $file = $this->StripRepoPathFrom( substr($file, 2) )) {
            if (!$onlyActions or strpbrk($action, $onlyActions)) {
              $files[] = $file;
            }
          }
        }

      return $files;
    }

      function StripRepoPathFrom($file) {
        $root = $this->RepoRoot();
        if ($root !== $this->path) {
          $path = substr($this->path, strlen($root)).'/';
          if (substr($file, 0, strlen($path)) === $path) {
            $file = substr($file, strlen($path));
          } else {
            return null;
          }
        }

        return ltrim($file, '\\/');;
      }

      function RepoRoot() {
        static $prefix = 'Repository Root:';

        if ($this->repoRoot === null) {
          $info = $this->ExecSVN('info', $this->path.'@HEAD');
          foreach ($info as $line) {
            if (stripos($line, $prefix) === 0) {
              $this->repoRoot = rtrim(trim( substr($line, strlen($prefix)) ), '\\/');
            }
          }

          if ($this->repoRoot === null) {
            throw new OtherTaskError('Cannot get repository URL from `svn info`.');
          }
        }

        return $this->repoRoot;
      }

    function UpdatedFiles($destInterface) {
      return $this->FilesChangedInLastCommit('to determine updated files', 'M');
    }

    function RemovedFiles($destInterface) {
      return $this->FilesChangedInLastCommit('to determine removed files', 'D');
    }

    function NewFiles($destInterface) {
      return $this->FilesChangedInLastCommit('to determine added files', 'A');
    }

    function CleanUp() {
      if ($this->localPath) {
        RmDirRecursive($this->localPath);
        $this->localPath = null;
      }
    }
  }
