<?php

  require_once 'utils.php';
  require_once 'exceptions.php';
  require_once 'interfaces.php';
  require_once 'multilang-build-task.php';
  require_once 'build-task.php';
  require_once 'info-tasks.php';

class UWikiTask {
  static $arrayOptions = array('list-handlers', 'list-replacements', 'list-handlers',
                               'load', 'entities', 'entities',
                               'term', 'style-aliases', 'style-aliases',
                               'anchorize', 'include', 'config-path', 'copy',
                               'auto-anchor-contractions', 'language', 'languages');
  static $tasks = array();
  static $interfaces = array('source' => array(), 'destination' => array());

  public $flags, $options, $byIndex;
  public $progressCallback;  // function ($total, $current, $info, $this)

  function __construct($cl, $runNow = true) {
    $this->SetCL($cl);
    $this->options['chdir'] and chdir( $this->options['chdir'] );

    $runNow and $this->Run();
  }

  function SetCL($cl) {
    $cl = self::ParseCL($cl, self::$arrayOptions);
    self::NormalizeCL($cl);

    $this->flags = $cl['flags'];
    $this->options = $cl['options'];
    $this->byIndex = $cl['byIndex'];
  }

    // will ensure options in $arrayOpts ('name', 'name', ...) are arrays.
    static function ParseCL($args, $arrayOpts = array()) {
        $arrayOpts = array_flip($arrayOpts);

      $flags = array();
      $options = array();
      $byIndex = array();

      foreach ($args as $i => &$arg) {
        if ($arg !== '') {
          if ($arg[0] == '-') {
            if ($arg === '--') {
              $byIndex = array_merge($byIndex, array_slice($args, $i + 1));
              break;
            } elseif ($argValue = ltrim($arg, '-')) {
              if ($arg[1] == '-') {
                @list($key, $value) = explode('=', $argValue, 2);
                $key = strtolower($key);
                isset($value) or $value = true;

                if (preg_match('/^(.+)\[(.*)\]$/', $key, $matches)) {
                  list(, $key, $subKey) = $matches;
                } else {
                  $subKey = null;
                }

                if ($subKey !== null and isset( $arrayOpts[$key] )) {
                  $subKey === '' ? $options[$key][] = $value
                                 : $options[$key][$subKey] = $value;
                } else {
                  isset( $arrayOpts[$key] ) and $value = array($value);
                  $options[$key] = $value;
                }
              } else {
                $flags += array_flip( str_split($argValue) );
              }
            }
          } else {
            $byIndex[] = $arg;
          }
        }
      }

      $flags and $flags = array_combine( array_keys($flags), array_fill(0, count($flags), true) );
      return compact('flags', 'options', 'byIndex');
    }

    static function NormalizeCL(&$cl) {
      $cl['flags']['q'] and $cl['flags']['y'] = true;
      IsEmptyStr( $cl['options']['markup'] ) and $cl['options']['markup'] = 'wacko';
      IsEmptyStr( $cl['options']['svn-command'] ) and $cl['options']['svn-command'] = 'svn';

      foreach (array('in-charset', 'out-charset') as $optName) {
        $charset = $cl['options'][$optName] = strtoupper( $cl['options'][$optName] );
        if (in_array($charset, array('UTF', 'UTF8', 'UTF-8'))) { unset( $cl['options'][$optName] ); }
      }
    }

  function Run() {
    if (is_array( $this->options['include'] )) {
      foreach ($this->options['include'] as $_file) { include_once $_file; }
    }

    $this->InitializeTask();
    $this->CheckEnvironment();
    $this->task->Run();
  }

    function InitializeTask() {
      $this->task = null;

        foreach (self::$tasks as $class) {
          if (call_user_func(array($class, 'CanHandle'), $this)) {
            $this->task = new $class($this);
            break;
          }
        }

      if (!$this->task) { throw new EUnknownTask(); }
    }

    function CheckEnvironment() {
      $errors = array(
        'PHP version is too low, v4 required, v5 recommended.' => version_compare(PHP_VERSION, '4.0.0', '<'),
        'mb_string PHP extension must be installed.' => !function_exists('mb_internal_encoding'),
        'iconv PHP extension must be installed if using --in-charset or --out-charset.' =>
          ($this->options['in-charset'] or $this->options['out-charset']) and !function_exists('iconv')
      );

      EEnvironmentCheck::ThrowIfAny($errors);
    }

  // $current - 1-based (last $current = $total). If returns true the caller might want to cancel current operation.
  function StopOnProgress($total, $current, $info) {
    if (is_callable($this->progressCallback)) {
      return call_user_func($this->progressCallback, $total, $current, $info, $this);
    }
  }
    function Progress($total, $current, $info) { $this->StopOnProgress($total, $current, $info); }
    function ProgressMessage($message) { $this->Progress(1, 1, $message); }

  function Render($text, $srcMarkup, $destFormat) {
    $doc = $this->CreateUWikiDocument($text, $srcMarkup, $destFormat);
    $doc->Parse();
    return $doc->RenderIn($destFormat);
  }

    function CreateUWikiDocument($text, $loadMarkup = null, $destFormat = null) {
      if ($path = &$this->options['uwiki-path']) {
        $path = rtrim(realpath($path), '\\/');
      } else {
        $path = '.';
      }

      defined('UWikiRootPath') or define('UWikiRootPath', $path);
      include_once "$path/uversewiki.php";

        if (!class_exists('UWikiDocument')) { throw new OtherTaskError('Cannot load UverseWiki class (usually in uversewiki.php).'); }

      $doc = new UWikiDocument('');
      IsEmptyStr($loadMarkup) or $doc->LoadMarkup($loadMarkup);
      $this->SetUWikiSettingsTo($doc, $destFormat);
      $doc->SetSource($text);

      return $doc;
    }

      // null $destFormat means it's unknown.
      function SetUWikiSettingsTo($doc, $destFormat = null) {
        static $configurable = array('DraftMode', 'NovelMode', 'styleAliases',
                                     'anchorize', 'expandTerms', 'dubiousAsComments',
                                     'noLineBreaksInsideParagraph', 'moreIndentBreaksParaLine',
                                     'inlineAnchors', 'inlineFootnotesAs',
                                     'footnotesAtTheEnd', 'unindentBlocks', 'wackoWikiHighlDir',
                                     'linkExt', 'fastTextFormatting', 'autoAnchorContractions',
                                     'headingMode');

          $options = &$this->options;
          $settings = &$doc->settings;

        $settings->hideDocTitle = true;
        $settings->wordCharacters .= $options['letters'];

        if ($url = $options['root-url']) { $settings->rootURL = rtrim($url, '\\/').'/'; }
        if ($url = $options['base-url']) { $settings->BaseURL($url); }
        if ($url = $options['media-url']) { $settings->mediaURL = rtrim($url, '\\/').'/'; }
        if ($url = $options['smiley-url']) { $settings->smileyURL = rtrim($url, '\\/').'/'; }
        if ($enableHTML5 = $options['enable-html5']) { $settings->enableHTML5 = $enableHTML5; }

        foreach ($configurable as $name) {
          $value = $options[ strtolower(ltrim( preg_replace('/[A-Z]/', '-\0', $name), '-' )) ];

          if (isset($value)) {
            if (strtoupper($name[0]) === $name[0]) {
              $settings->$name($value);  // it's a mode.
            } else {
              $setting = &$settings->$name;
              $setting = is_array($setting) ? array_merge($setting, $value) : $value;
            }
          }
        }

        if (is_array($options['load'])) {
          foreach ($options['load'] as $file) {
            $doc->LoadModuleInto($settings->markupName, $file);
          }
        }

        if (is_array($options['entities'])) {
          foreach ($options['entities'] as $name => $entity) {
            list($code, $html) = explode(';', $entity, 2);

            if (is_numeric($code) and !IsEmptyStr($html)) {
              $code = (int) $code;
              $settings->entities[$name] = compact('code', 'symbol');
            } else {
              unset( $settings->entities[$name] );
            }
          }
        }

        if (is_array($options['term'] and $settings->markupName === 'wacko')) {
          foreach ($options['term'] as $term => $meaning) {
            $settings->FloodReplaceTerm($term, $meaning);
          }
        }

        if (empty( $options['config-path'] )) {
          $settings->LoadFrom(UWikiRootPath.'/config');
        } else {
          foreach ($options['config-path'] as $path) { $settings->LoadFrom($path); }
        }
      }
}

abstract class BaseTask {
  public $cl;

  abstract static function CanHandle($cl);

  function __construct($cl) { $this->cl = $cl; }
  function CheckEnvironment() { }
  abstract function Run();
}

class UWikiTemplater {
  public $tplPath, $preparedTplFile, $tplFile, $cl;

  function __construct($tplPath, $cl) {
    $this->cl = $cl;
    $this->SetPath($tplPath);
  }

    function SetPath($tplPath) {
      $this->tplPath = rtrim($tplPath, '\\/');

      if (IsEmptyStr( $cl->options['page'] )) {
        $tplFile = glob( $this->tplPath.'/_page.{html,htm,php,txt,wiki,wacko}', GLOB_BRACE );
        $tplFile = array_shift($tplFile);
        $this->tplFile = $tplFile ? $tplFile : '_page.html';
      } else {
        $this->tplFile = $this->tplPath.'/'.$cl->options['page'];
      }

      $this->CleanUp();
    }

    function GetPath() { return $this->tplPath; }

  function CheckEnvironment() {
    $errors = array(
      'Template directory does not exist.' => !is_dir($this->tplPath) or !is_readable($this->tplPath),
      'Page template file doesn\'t exist.' => !is_file($this->tplFile) or !is_readable($this->tplFile),
    );

    EEnvironmentCheck::ThrowIfAny($errors);
  }

  function CleanUp() {
    if ($this->preparedTplFile !== null) {
      unlink($this->preparedTplFile);
      $this->preparedTplFile = null;
    }
  }

  function FillWith($vars) {
    $this->PreparePageTpl();
    return $this->FillPrepared($this->preparedTplFile, $vars);
  }

    function PreparePageTpl() {
      if ($this->preparedTplFile === null) {
        $this->preparedTplFile = tempnam(null, '');
        file_put_contents( $this->preparedTplFile, $this->PrepareTPL($this->tplFile) );
      }
    }

  function FillInPlace($file, $vars) { return $this->FillFileInto($file, $file, $vars); }

  function FillFile($tplFile, $vars) {
    $dest = tempnam(null, '');
    if ($this->FillFileInto($dest, $tplFile, $vars) !== null) {
      $filled = file_get_contents($dest);
      unlink($dest);
      return $filled;
    }
  }

  function FillFileInto($destFile, $tplFile, $vars) {
    file_put_contents($destFile, $this->PrepareTpl($tplFile));

    try {
      return file_put_contents($destFile, $this->FillPrepared($destFile, $vars));
    } catch (Exception $e) {
      $destFile === $tplFile or unlink($destFile);
      throw $e;
    }
  }

    function PrepareTpl($file) {
      $script = file_get_contents($file);

      if ($this->cl->options['static']) {
        return strtr(preg_replace('~<\?\?.*?\?>(\r?\n)*~su', '', $script),
                     array('<?%' => '<?'));
      } else {
        return strtr($script, array('<??' => '<?php echo \'<?\'?>',
                                    '<?%' => '<?php echo \'<?\'?>'));
      }
    }

  function &FillPrepared($_file, &$_vars) {
    is_array($_vars) and extract($_vars, EXTR_SKIP);
    isset($cl) or $cl = $this->cl;
    if (!isset($strings) and isset($doc)) { $strings = $doc->settings->strings; }

    ob_start();
      $included = include($_file);
    $output = ob_get_clean();

      if ($included === false or IsEmptyStr($output)) {
        throw new OtherTaskError("Formatting of template $_file fhas failed.");
      }

    return $output;
  }
}
