<?php

class BComments {
  // settings for Remember/RecallIntoTo():
  static $fieldsToRemember = array('author', 'authorEMail', 'authorHome', 'signatureSource');
  static $cookieGroup = 'blog-comments';   // actual cookie name will be $cookieGroup[name].
  static $cookieExpireTime = 2592000;      // seconds; default is 30 days.

  static function Format($text) {
    $event = 'format comment: '.BConfig::$commentMarkup;
    if (!BEvent::Exists($event)) {
      throw new BException('Comment markup "'.BConfig::$commentMarkup.'" is not installed.');
    }

    return BEvent::Fire($event, array(&$text));
  }

  // 0 limit = unlimited. $limits other than #0 are used to limit subcomment depth.
  static function Of($parent, $options = array()) {
    $options += array('limits' => array(0, 10, 5), 'currentPage' => 0,
                      'fullVars' => true, 'sort' => null);
    extract($options, EXTR_SKIP);

    $comments = array();

    if ($limits) {
      $limit = array_shift($limits);
      $post = self::PostOf($parent);

      $files = glob( self::FileOf($parent.'/*') );
      $files = array_slice($files, $limit * $currentPage);

      foreach ($files as $file) {
        $comment = $parent.'/'.basename($file, '.php');

        $vars = self::GetVarsOf($comment, $limits, $fullVars, $sort);
        $comments[] = $vars;

        if (--$limit === 0) { break; }
      }
    }

    self::SortAs($sort, $comments);
    return $comments;
  }

  static function CommentedPosts() {
    $result = array();

      foreach (scandir(BConfig::FileOf('comments')) as $file) {
        is_dir($file) and $result[] = self::PostOf($file);
      }

    return $result;
  }

  static function AllRecursive() {
    $result = array();

      $root = BConfig::FromUTF8( 'file name', BConfig::$paths['comments'] );
      foreach (scandir($root) as $file) {
        if ($file[0] !== '.' and is_dir($root.$file)) {
          try {
            $post = self::PostOf($file);
            BPosts::Exists($post) and $result[$post] = self::AllRecursiveOf($file, $root);
          } catch (Exception $e) { }
        }
      }

    return $result;
  }

    protected static function AllRecursiveOf($path, $root) {
      $result = array();

        foreach (scandir($root.$path) as $base) {
          if ($base[0] !== '.') {
            $file = "$path/$base";
            if (is_file($root.$file) and ExtOf($base) === '.php') {
              $result[] = substr($file, 0, -4);
            } elseif (is_dir($root.$file)) {
              $result = array_merge($result, self::AllRecursiveOf($file, $root));
            }
          }
        }

      return $result;
    }

  static function SortAs(&$way, &$comments) {
    if ($way) {
      if (method_exists(__CLASS__, 'SortBy'.$way)) {
        usort($comments, array(__CLASS__, 'SortBy'.$way));
      } else {
        BEvent::Fire('sort comments: '.$way, array(&$comments));
      }
    }
  }

    static function SortByTimeAsc($first, $second) {
      $f = $first['time'];
      $s = $second['time'];
      return ( $f > $s ? +1 : ($f < $s ? -1 : 0) );
    }

    static function SortByTimeDesc($first, $second) {
      $f = $first['time'];
      $s = $second['time'];
      return ( $f > $s ? -1 : ($f < $s ? +1 : 0) );
    }

  static function Total() { return count( BIndex::SelectFrom('CommentDates') ); }
  static function TotalOf($post) { return BIndex::SelectFrom('CommentCounts', $post); }

    static function TotalOfComment($comment) {
      $total = 0;

        $sub = glob( self::FileOf($comment, false).'/*' );
        foreach ($sub as $file) { is_file($file) and ++$total; }

      return $total;
    }

  static function PostOf($comment) { return self::UnmaskPost( strtok($comment, '/') ); }
  static function BelongsToPost($comment) { return strpos($comment, '/') === false; }

  static function CommentOf($comment) {
    strtok($comment, '/');
    return strtok(null);
  }

    // returns null if comment belongs directly to a post.
    static function ParentCommentOf($comment) {
      $start = strpos($comment, '/');
      $end = strrpos($comment, '/', $start + 1);
      if ($start and $end) {
        return substr($comment, 0, $end);
      }
    }

    // purpose of masking is to have comment name structure of: parent-post/com_1/com_1.1/...
    // so: (1) post name must not represent several paths (parent/post), (2) have no chars
    //     that are special in file names, (3) is recommented to be charset-safe
    //     (for transmission via web forms, etc.) - i.e. use only ACSII chars.
    static function MaskPost($str) {
      return BEvent::FireResult('string', 'mask post', $str);
    }

    static function UnmaskPost($str) {
      return BEvent::FireResult('string', 'unmask post', $str);
    }

  // $comment: either array($post, $comment) or just $comment (with already encoded post).
  static function FileOf($comment, $ext = '.php') {
    is_array($comment) and $comment = self::ReferrerFor($comment[0], $comment[1]);
    return BConfig::FileOf('comments', $comment.$ext);
  }

    static function Exists($comment) { return is_file( self::FileOf($comment) ); }

    static function ReferrerFor($post, $comment = null) {
      $post = self::MaskPost($post);
      return $comment === null ? $post : "$post/$comment";
    }

  // $parent must be a referrer string.
  static function RenameParent($parent, $newName) {
    if (!rename($old = BConfig::FileOf('comments', $parent),
                $new = BConfig::FileOf('comments', $newName))) {
      throw new BException("Cannot rename group of comments $old -> $new.");
    }
  }

    static function PostMoved($post, $newName) {
      $post = self::ReferrerFor($post);
      if (is_dir( self::FileOf($post, false) )) {
        self::RenameParent($post, self::ReferrerFor($newName));
      }
    }

  static function NextTo($comment) { return self::GetByOffsetTo($comment, +1); }
  static function PreviousTo($comment) { return self::GetByOffsetTo($comment, -1); }
  static function First() { return self::GetByOffsetTo(null, 'first'); }
  static function Last() { return self::GetByOffsetTo(null, 'last'); }
  static function Random() { return self::GetByOffsetTo(null, 'random'); }

    static function GetByOffsetTo($comment, $offset, $index = 'CommentDates') {
      return BIndex::SelectFrom($index, $comment, $offset);
    }

  static function NextOf($post, $comment) { return self::GetByOffsetTo(array($post, $comment), +1, 'PostCommentDates'); }
  static function PreviousOf($post, $comment) { return self::GetByOffsetTo(array($post, $comment), -1, 'PostCommentDates'); }
  static function FirstOf($post) { return self::GetByOffsetTo($post, 'first', 'PostCommentDates'); }
  static function LastOf($post) { return self::GetByOffsetTo($post, 'last', 'PostCommentDates'); }
  static function RandomOf($post) { return self::GetByOffsetTo($post, 'random', 'PostCommentDates'); }

  // returns -1 for post ("post-file").
  static function DepthOf($comment) {
    return substr_count($comment, '/') - 1;   // "post-file/comment /deeper comment/..."
  }

  static function UrlOf($comment) {
    // A comment can appear as (also take paging ($commentsPerPage) into account):
    // 1. Part of blog entry's page - as root comment;
    // 2. Again part of entry page but as nesting comment;
    // 3. Ultimately, a parent comment on its thread page.

    $page = self::PageIndexOf($comment);
    $page = $page ? 'page='.$page : '';

    $maxDepth = count(BConfig::$commentNestingLimits);
    $depth = self::DepthOf($comment);
    if ($depth === 0 or $depth <= $maxDepth) {
      $page and $page = "?$page";
      $anchor = '#comment'.self::Get($comment, 'id');
      return BPosts::Urlof( self::PostOf($comment) ).$page.$anchor;
    } else {
      return self::ThreadUrlOf($comment, $page);
    }
  }

    static function PageIndexOf($comment) {
      if (BConfig::$commentsPerPage === 0) {
        return 1;
      } else {
        $siblings = glob( self::FileOf(dirname($comment).'/*') );
        $i = array_search( self::FileOf($comment), $siblings );

          if ($i === false) {
            throw new BException('Cannot get page index of comment '.$comment.' as it doesn\'t exist.');
          }

        return (int) ($i / BConfig::$commentsPerPage);
      }
    }

    static function ThreadUrlOf($comment, $page = '') {
      $page and $page = "&$page";
      return BPosts::Urlof( self::PostOf($comment) ).'/thread?parent='.
             urlencode( self::CommentOf($comment) ).$page;
    }

  static function Get($comment, $field = null) {
    $file = self::FileOf($comment);
    if (!$file) {
      throw new BException('Comment '.$comment.' doesn\'t exist.');
    }

    $vars = (include($file));
    if (!$vars or !is_array($vars)) {
      throw new BException('Failed to load comment.', $file);
    }

    $vars['comment'] = $comment;
    $vars['isByAnonymous'] and $vars['author'] = Translate('Anonymous');

    return $field === null ? $vars : $vars[$field];
  }

  static function GetVarsOf($comment, $subcommLimits, $fullVars, $sort = null) {
    $vars = self::Get($comment);

    $sub = $vars['subcomments'] = self::GetSubcommentsOf($comment, $subcommLimits, $fullVars, $sort);
    $vars['moreSubcomments'] = max( 0, self::TotalOfComment($comment) - count($sub) );

    $fullVars and BTplVars::Set('thread', $vars, $comment);
    return $vars;
  }

    static function GetSubcommentsOf($comment, $limits, $fullVars, $sort = null) {
      $sub = array();

      if ($limits) {
        $limit = array_shift($limits);
        $files = glob( self::FileOf("$comment/*", false) );

        foreach ($files as $file) {
          if (is_file($file)) {
            $file = basename($file, '.php');
            $sub[] = self::GetVarsOf("$comment/$file", $limits, $fullVars, $sort);
            if (--$limit === 0) { break; }
          }
        }
      }

      self::SortAs($sort, $sub);
      return $sub;
    }

  static function AddTo($parent, $info, $comment = null) {
    if (!BPosts::Exists( self::PostOf($parent) ) and !self::Exists($parent)) {
      throw new BException('Post or comment to add comment to ('.$parent.') didn\'t exist.');
    }

    BEvent::Fire('normalize comment', array(&$parent, &$comment, &$info));
    $comment === null and $comment = self::Save($info, $parent);
    BEvent::Fire('comment added', array($parent, $comment, &$info));

    return $comment;
  }

    static function ApplyMaxDepth(&$parent, $comment, &$info) {
      if ($max = BConfig::$maxCommentNesting) {
        while (self::DepthOf($parent) >= $max - 1) {
          $top = self::ParentCommentOf($parent);
          $parent = $top ? $top : self::ReferrerFor( self::PostOf($parent) );
        }
      }
    }

    static function Normalize($parent, $comment, &$info) {
      // don't ;trim because of possible indentation (e.g. "  * list").
      foreach ($info as &$field) { $field = rtrim((string) $field); }

      if (((string) $info['text']) === '') {
        throw new BException('Comment must have non-empty text field.');
      }

      $info += array('author' => '', 'authorHome' => '', 'authorEMail' => '',
                     'signature' => '', 'time' => time());


        $info['ip'] = $_SERVER['REMOTE_ADDR'];
        $info['author'] = trim($info['author']);
        $info['authorEMail'] = trim($info['authorEMail']);

        $home = trim($info['authorHome']);
        if ($home !== '' and strpos($home, '//') === false) {
          $info['authorHome'] = 'http://'.$info['authorHome'];
        }

        $anonStr = Translate('Anonymous');
        $info['author'] === '' and $info['author'] = $anonStr;
        $info['isByAnonymous'] = mb_strtolower($info['author']) === mb_strtolower($anonStr);

        if ($parent === null) {
          $info['isByPostAuthor'] = false;
        } else {
          $postAuthor = mb_strtolower( BPosts::Get(self::PostOf($parent), 'author') );
          $info['isByPostAuthor'] = mb_strtolower($info['author']) === $postAuthor;
        }

        $info['formatter'] = BConfig::$commentMarkup;

        $info['source'] = $info['text'];
        $info['text'] = self::Format($info['text']);
        $info['signatureSource'] = $info['signature'];
        $info['signature'] = self::Format($info['signature']);
    }

    static function Save(&$info, $parent) {
      $dir = self::FileOf($parent, false);
      EnsureDirExists($dir);

        if (BConfig::$consistentCommentIDs) {
          $files = glob("$dir/*");
          foreach ($files as &$file) { $file = (int) basename($file, '.php'); }
          $files = array_filter($files);
          sort($files);
          $i = array_pop($files);
        } else {
          $i = count( glob("$dir/*") );
        }

      do { $file = "$dir/".++$i.'.php'; } while (file_exists($file));

      if (self::Exists($parent)) {
        $parentID = self::Get($parent, 'id');
        $info['id'] = "$parentID.$i";
      } else {
        $info['id'] = $i;
      }

      // this is commenting form var which shouldn't be used to identify parent post/comment
      // because it isn't changed upon RenameParent (use PostOf, ParentCOmmentOf and other methods).
      unset($info['post']);
      self::SaveTo($file, $info);

      return "$parent/$i";
    }

      static function SaveTo($file, $info) {
        $export = "<?php\nreturn ".var_export($info, true).';';
        if (!is_int( file_put_contents($file, $export) )) {
          throw new BException('Cannot write comment file '.$file);
        }
      }

  static function Update($comment, $info) {
    if (!self::Exists($comment)) {
      throw new BException('Comment to update '.$comment.' didn\'t exist.');
    }

    $parent = self::ParentCommentOf($comment);

    BEvent::Fire('normalize comment', array(&$parent, &$comment, &$info));
    self::SaveTo(self::FileOf($comment), $info);
    BEvent::Fire('comment updated', array($parent, $comment, &$info));
  }

  static function Delete($comment) {
    if (!self::Exists($comment)) {
      throw new BException('Comment to delete '.$comment.' didn\'t exist.');
    }

    $info = self::Get($comment);
    BEvent::Fire('before deleting comment', array($comment, &$info));
    unlink(self::FileOf($comment));
    BEvent::Fire('comment deleted', array($comment, &$info));
  }

  // todo: should also RenameParent (move subcomments) and check if $newName isn't thread of itself ($comment).
  static function Move($comment, $newName) {
    if (!self::Exists($comment)) {
      throw new BException('Comment to move '.$comment.' didn\'t exist.');
    }
    if (self::Exists($comment)) {
      throw new BException('New name "'.$newName.'" for comment '.$comment.' is already registered.');
    }

    $info = self::Get($comment);
    BEvent::Fire('before moving comment', array($comment, &$newName, &$info));

    rename($old = self::FileOf($comment), $new = self::FileOf($newName));
    if (!self::Exists($newName)) {
      throw new BException("Cannot rename comment file $old -> $new.");
    }

    BEvent::Fire('comment moved', array($comment, $newName, &$info));
  }

  static function AvatarFor($comment, $size) {
    $type = BConfig::$avatars;

    if ($type and $type !== 'off') {
      $event = 'avatar: '.$type;
      if (!BEvent::Exists($event)) {
        throw new BException('Don\'t know how to get "'.$type.'" avatar.');
      }

      $info = self::Get($comment);
      return BEvent::Fire($event, array($comment, &$info, $size));
    }
  }

    static function Gravatar($comment, &$info, $size) {
      $ident = $info['authorEMail'];
      $ident or $ident = $info['authorHome'];
      if (!$ident and !$info['isByAnonymous']) {
        $ident = $info['author'];
      }
      $ident or $ident = uniqid(mt_rand(), true);

      $vars = array('{HASH}' => md5($ident), '{SIZE}' => $size);
      return strtr( trim(BConfig::$paths['avatars'], '\\/'), $vars );
    }

  function RememberInfoIfAdded($parent, $comment, &$info) {
    self::Exists($comment) or self::RememberInfoFrom($info);
  }

    function RememberInfoFrom(&$info) {
      foreach (self::$fieldsToRemember as $name) {
        setcookie(self::$cookieGroup."[$name]", $info[$name], time() + self::$cookieExpireTime, '/');
      }
    }

  function RecallInfo($quoteHTML = false) {
    $info = array();

      foreach (self::$fieldsToRemember as $name) {
        $value = &$_COOKIE[self::$cookieGroup][$name];
        $info[$name] = trim($value);
        $quoteHTML and $info[$name] = htmlspecialchars8( $info[$name] );
      }

    $info['signature'] = &$info['signatureSource'];
    return $info;
  }
}
