<?php

abstract class UWikiPager {
  // in place of $format parameter for ReadPage(), etc.
  const AutoFormat = null;

  // return values of PageExists() and other similar functions.
  const Unknown = null;
  const NotFound = false;

  const MaxFormatNameLength = 8;

  // either a file name (<cluster>/index.wacko) or a string ".." (<cluster>.wacko or
  // "/index.wacko" for root cluster index).
  public $indexPage = 'index';
  public $indexFormat = 'wacko';

  // used for accessing files having no (default) extensions, e.g. in case of ((links)) -> links.wacko.
  public $defaultFormat = 'wacko';

  protected $current;

  function __construct() { $this->SetCurrent('index'); }

  function SetCurrent($current, $format = null) {
    if ($format !== null and substr($current, -1 * strlen($format) - 1) !== ".$format") {
      $current .= ".$format";
    }

    "$current" === '' and $current = "/{$this->indexPage}.{$this->indexFormat}";
    $current = $this->ExpandLocal($current);

      if ($this->ClusterExists($current)) {
        $format === null and $format = $this->FormatFrom($current, $this->indexFormat);
        $current .= "/{$this->indexPage}.$format";
      }

    $this->current = ltrim($current, '\\/');
  }

  function GetCurrent() { return '/'.$this->current; }

  function ExpandLocal($path) {
    return $this->Expand($path, dirname($this->GetCurrent()));
  }

  function IsCurrent($page) {
    $current = $this->GetCurrent();

    $page = $this->ExpandLocal($page);
    if ($this->FormatFrom($page, false) === false) {
      $page .= '.'.$this->defaultFormat;
    }

    $paths = array(&$page, &$current);
    foreach ($paths as &$str) {
      $existing = $this->FormatFrom($str, false);
      $real = $this->RealFormatBy(self::AutoFormat, $str);
      if ($real !== $existing) {
        $str = ($existing === false ? "$str." : substr($str, 0, -1 * strlen($existing))).$real;
      }
    }

    return '/'.ltrim($page, '\\/') === $current;
  }

  // 3 methods below return null if unknown (e.g. outside base path or format is not allowed),
  // false if could access but resource didn't exist.
  // RealFormatBy() is called if (string) $format === ''.
  abstract function ReadPage($page, $format = self::AutoFormat);
  abstract function PageExists($page, $format = self::AutoFormat);
  abstract function ClusterExists($page);
  abstract protected function ReadCluster($cluster);
  abstract function PageInfo($page);
  abstract function ClusterInfo($page);

  // returns a string, an UWikiBaseElement or null - see UWikiDocument::GetTitleOf().
  function GetTitleOf($fileOrCluster, $format = self::AutoFormat) {
    if ($this->ClusterExists($fileOrCluster)) {
      return $this->GetClusterTitle($fileOrCluster);
    } else {
      return $this->GetPageTitle($fileOrCluster, $format);
    }
  }

    function GetPageTitle($page, $format = self::AutoFormat) {
      $contents = $this->ReadPage($page, $format);
      if (is_string($page)) {
        return UWikiDocument::GetTitleOf($contents, $this->RealFormatBy($format, $page));
      } else {
        return '';
      }
    }

    function GetClusterTitle($cluster) {
      $indexPage = $this->IndexPageNameOf($cluster);
      return $this->GetPageTitle($indexPage, $this->indexFormat);
    }

      function IndexPageNameOf($cluster, $withFormat = false) {
        $cluster = rtrim($cluster, '\\/');

        if ($this->indexPage === '..') {
          static $roots = array('' => 1, '.' => 1, '..' => 1, '/' => 1);
          $result = isset($roots[$cluster]) ? 'index' : $cluster;
        } else {
          $result = "$cluster/".rtrim($this->indexPage, '\\/');
        }

        $withFormat and $result .= '.'.$this->indexFormat;
        return $result;
      }

  // 'u' (Unicode) modifier is automatically added for all regexps ('*RE').
  // other $options are: "prop=" => "value", "prop!=" => "value".
  function ListCluster($cluster, $options = array(), $prefix = '') {
    $options += array('match' => array(), 'ignore' => array(), 'matchRE' => array(),
                      'ignoreRE' => array(), 'matchFuncs' => null, 'ignoreFuncs' => null,
                      'only' => array(), 'not' => array(), 'recursive' => false,
                      'sort' => true, 'ignoreByDefault' => false);
    $isRecursive = $options['recursive'];

    $result = array();

      $entries = $this->ReadCluster($cluster);
      if (is_array($entries)) {
        foreach ($entries as $baseName) {
          $name = rtrim($cluster, '\\/')."/$baseName";

          $entry = compact('name');
          $entry += (array) ($this->ClusterExists($name) ? $this->CLusterInfo($name)
                                                         : $this->PageInfo($name));

          if ($this->MatchEntryAgainst($options, $entry)) {
            $result[$prefix.$baseName] = $entry;
            $isRecursive and $result += $this->ListCluster($name, $options, $name.'/');
          }
        }
      }

    if ($sorter = $options['sort']) {
      is_callable($sorter) ? uasort($result, $sorter) : ksort($result);
    }

    return $result;
  }

    function MatchEntryAgainst($options, $file) {
      extract($options, EXTR_SKIP);

      if (!empty($match) and $this->MatchWildcards($match, $file['name'])) { return true; }
      if (!empty($matchRE) and $this->MatchRegExps($matchRE, $file['name'])) { return true; }
      if (!empty($matchFuncs) and $this->MatchCallbacks($matchFuncs, $file, $options)) { return true; }
      if (!empty($only) and $this->MatchEntryTypes($only, $file)) { return true; }
      if ($this->MatchPropsEqualityOf($file, $options, '=')) { return true; }

      if (!empty($ignore) and $this->MatchWildcards($ignore, $file['name'])) { return false; }
      if (!empty($ignoreRE) and $this->MatchRegExps($ignoreRE, $file['name'])) { return false; }
      if (!empty($ignoreFuncs) and $this->MatchCallbacks($ignoreFuncs, $file, $options)) { return false; }
      if (!empty($not) and $this->MatchEntryTypes($not, $file)) { return false; }
      if ($this->MatchPropsEqualityOf($file, $options, '!=')) { return false; }

      return empty($options['ignoreByDefault']);
    }

      function MatchWildcards($wildcards, $fileName) {
        $wildcards = (array) $wildcards;
        foreach ($wildcards as $wildcard) {
          if (fnmatch($wildcard, $fileName)) { return true; }
        }
      }

      function MatchRegExps($regexps, $fileName) {
        $regexps = (array) $regexps;
        foreach ($regexps as $regexp) {
          if (preg_match($regexp.'u', $fileName)) { return true; }
        }
      }

      function MatchCallbacks($callbacks, $file, &$options) {
        if (is_array($callbacks)) {
          return UWikiDocument::CallAll($callbacks, array($file, $options, $this));
        }
      }

      function MatchEntryTypes($types, $entry) {
        $types = (array) $types;
        foreach ($types as $type) {
          if (!empty( $entry['is'.ucfirst($type)] )) { return true; }
        }
      }

      function MatchPropsEqualityOf($file, $options, $suffix) {
        $sufLen = strlen($suffix);
        foreach ($options as $name => $against) {
          if (substr($name, -1 * $sufLen) === $suffix and
              $file[ substr($name, 0, -1 * $sufLen) ] == $against) {
            return true;
          }
        }
      }

  function RealFormatBy($format, $page) {
    $format = (string) $format;
    if ($format === '' or $format === self::AutoFormat) {
      $format = $this->FormatFrom($page);
    }
    return strtolower($format);
  }

    function FormatFrom($pageName, $defaultFormat = null) {
      $result = null;

        $tail = mb_substr($pageName, -1 * self::MaxFormatNameLength);
        $pos = mb_strrpos($tail, '.');
        if ($pos !== false and isset($pageName[$pos + 1])) {
          $format = mb_substr($tail, $pos + 1);
          if (strpbrk($format, '\\/') !== false) { $format = null; }
        }

      return (isset($format) ? $format : ($defaultFormat === null ? $this->defaultFormat : $defaultFormat));
    }

  // path w/o trailing slash unless it's just "/" or "c:/"; converts \ to / removing
  // successive / and \; resolves '.' and '..' relative to $cwd.
  static function Expand($path, $cwd = null) {
    $path = self::ExpandLeaveLinks($path, $cwd);
    return self::ReadLinkUntilReal($path);
  }

    static function ExpandLeaveLinks($path, $cwd = null) {
      if (!is_string($path)) { return; }

        $cwd === null and $cwd = getcwd();
        $cwd = rtrim($cwd, '\\/');

      $firstIsSlash = strpbrk(@$path[0], '\\/');
      if ($path === '' or (!$firstIsSlash and @$path[1] !== ':')) {
        $path = "$cwd/$path";
      } elseif ($firstIsSlash and @$cwd[1] === ':') {
        // when a drive is specified in CWD root \ or /) refers to its root.
        $path = substr($cwd, 0, 2).$path;
      }

      $path = strtr($path, '\\', '/');

        if ($path !== '' and ($path[0] === '/' or @$path[1] === ':')) {
          list($prefix, $path) = explode('/', $path, 2);
          $prefix .= '/';
        } else {
          $prefix = '';
        }

      $expanded = array();
      foreach (explode('/', $path) as $dir) {
        if ($dir === '..') {
          array_pop($expanded);
        } elseif ($dir !== '' and $dir !== '.') {
          $expanded[] = $dir;
        }
      }

      return $prefix.join('/', $expanded);
    }

    static function ReadLinkUntilReal($path) {
      if (function_exists('readlink')) {  // prior to PHP 5.3 it only works for *nix.
        while (is_link($path) and ($target = readlink($path)) !== false) { $path = $target; }
      }

      return $path;
    }
}

// This class is like /dev/null - accepting anything but returning nothing.
class UWikiPagerStub extends UWikiPager {
  function ReadPage($page, $format = self::AutoFormat) { }
  function PageExists($page, $format = self::AutoFormat) { }
  function ClusterExists($page) { }
  protected function ReadCluster($cluster) { }
  function PageInfo($page) { }
  function ClusterInfo($page) { }
}

class UWikiFilePager extends UWikiPager {
  // array of 'format' => true/false (null)/'path'/array(call, back), ...
  // * 'path' will change base dir for the format. Non-existing paths will deny access.
  // * callback will be called and if it returns true the file/dir will be read. It can
  //   return null if access was denied or false if it wasn't found. No base dir checks
  //   are made and file path isn't resolved to full path either so do it yourself.
  //   Format: function (&$page, $this); you can change $page.
  // Allow formats with caution as it may reveal server-only data, e.g. {{Include index, php}}.
  // Not listed formats here are denied from access. 'txt' & 'wiki' are aliases of 'wacko'
  // and can be controlled separately (but don't set different base dirs for them).
  // $allowedFormats can also be a true/false (null) value to allow/disallow global access.
  // There's a special format "*dir" that controls access to clusters (not files inside them).
  public $allowedFormats = array('*dir' => true, 'wacko' => true, 'txt' => true, 'wiki' => true);
  public $wackoAliases = array('wiki' => true, 'txt' => true);
  // a callback to be invoked on each file name being used - useful to convert UTF-8
  // names (since anything inside UWiki is in UTF-8) into charset supported by file
  // system (such as cp1251).
  public $nameConvertor;

  protected $path;

  function __construct($baseDir) {
    parent::__construct();
    $this->SetBaseDir($baseDir);
  }

    function SetBaseDir($newDir) {
      $baseDir = self::Expand($newDir);
      if (!is_dir($baseDir) or !is_readable($baseDir)) { throw new EUWikiUnreadableFilePagerFolder($newDir); }
      $this->path = rtrim($baseDir, '\\/');
    }

  // excluding trailing slash.
  function GetBaseDir() { return $this->path; }

  function ConvertFileName($name) {
    $cnv = $this->nameConvertor;
    return (string) ($cnv ? UWikiDOcument::Call($cnv, array(&$name, $this)) : $name);
  }

  // function ($page, $format) or ($page, true = $isDir)
  // $page is in UTF-8, result is ran thru $nameConvertor or in UTF-8 if it's undefined.
  function FileOf($page, $format, $isDir = false) {
    if ($format === true) {
      $isDir = true;
      $format = '*dir';
    } else {
      $format = $this->RealFormatBy($format, $page, true);
    }

      if (!$isDir and $format === 'wacko') {
        $data = $this->FileOf($page, 'txt');
        if ($data === self::NotFound or $data === self::Unknown) { $data = $this->FileOf($page, 'wiki');}
        if ($data !== self::NotFound and $data !== self::Unknown) { return $data; }
      }

    $page = str_replace('\\', '/', $page);
    if (!$isDir and substr($page, -1 * strlen($format) - 1) !== ".$format") {
      $page .= ".$format";
    }

      $page = $this->ConvertFileName($page);
      if ($page === '') { return self::NotFound; }

    $allow = $this->allowedFormats;
    if (is_array($allow) and !UWikiDocument::CanCall($allow)) {
      $allow = @$this->allowedFormats[$format];
    }

      if ($allow) {
        if (is_array($allow)) {
          $allow = UWikiDocument::Call($allow, array(&$page, $this));
        } else {
          $baseDir = $allow === true ? $this->path : self::Expand($allow, $this->path);

            if ($page[0] === '/') {
              $page = self::Expand(ltrim($page, '/'), $this->ConvertFileName($baseDir));
            } else {
              $relBaseDir = rtrim($baseDir, '/').'/'.dirname($this->GetCurrent());
              $page = self::Expand($page, $this->ConvertFileName($relBaseDir));
            }

          $allow = ($baseDir !== false and substr($page, 0, strlen($baseDir)) === $baseDir);
          $allow &= ($isDir or $page[ strlen($baseDir) ] === '/');
        }
      }

    if ($allow) {
      $allow = $isDir ? is_dir($page) : is_file($page);
      return $allow ? $page : self::NotFound;
    }
  }

    function RealFormatBy($format, $page, $noWackoAliasing = false) {
      $format = parent::RealFormatBy($format, $page);
      if (!$noWackoAliasing and !empty($this->wackoAliases[$format])) {
        $format = 'wacko';
      }
      return $format;
    }

  // false if doesn't exist, null if can't access, string otherwise. Same for others below.
  function ReadPage($page, $format = self::AutoFormat) {
    $file = $this->FileOf($page, $format);
    if ($file !== self::Unknown) { return is_file($file) ? file_get_contents($file) : self::NotFound; }
  }

  function PageExists($page, $format = self::AutoFormat) {
    $file = $this->FileOf($page, $format);
    if ($file !== self::Unknown) { return is_file($file) and is_readable($file); }
  }

  function ClusterExists($page) {
    $file = $this->FileOf($page, true);
    if ($file !== self::Unknown) { return is_dir($file) and is_readable($file); }
  }

  protected function ReadCluster($cluster) {
    if ($this->ClusterExists($cluster)) {
      $root = $this->FileOf($cluster, true).'/';
      $files = scandir($root);

        foreach ($files as $i => &$file) {
          $format = is_dir($root.$file) ? true : null;
          if ($file === '.' or $file === '..' or
              !$this->FileOf(rtrim($cluster, '\\/')."/$file", $format)) {
            unset($files[$i]);
          }
        }

      return $files;
    }
  }

  function PageInfo($page) { return $this->InfoAboutFile( $this->FileOf($page, null) ); }
  function ClusterInfo($page) { return $this->InfoAboutFile( $this->FileOf($page, true) ); }

    protected function InfoAboutFile($path) {
      if (is_string($path)) {
        $info = array('name' => $path, 'isDir' => is_dir($path), 'isFile' => is_file($path),
                      'isReadable' => is_readable($path), 'isWritable' => is_writable($path),
                      'isExecutable' => is_executable($path), 'isLink' => is_link($path),
                      'linkTarget' => is_link($path) ? self::ReadLinkUntilReal($path) : null,
                      'perms' => fileperms($path), 'gid' => filegroup($path),
                      'uid' => fileowner($path), 'modTime' => filemtime($path),
                      'type' => filetype($path), 'size' => filesize($path));

        return $info;
      }
    }
}

  class EUWikiUnreadableFilePagerFolder extends EUverseWiki {
    public $path;

    function __construct($path) {
      $this->path = $path;
      return parent::__construct("UverseWiki's File pager can't find directory $path or it's not readable.");
    }
  }
