<?php
  /* Always call UWikiSettings->LoadFrom() after you're done changing its props, not before. */

  define('UWikiLeftWordBoundary', '^|\.\.\.|[…\s"\'«“‘‛‹\/\\<[{(]');
  define('UWikiRightWordBoundary', '$|[…\s"\'»„”’‛›\/\\>\]})?!.,:;]');

class UWikiSettings {
  const CommentPrefixes = '#;';

  public $markupName;
  public $handlers;       // an instance of UWikiHandlerList.
  public $strings = array();
  public $pager;
  public $format;         // UWikiFormatSettings or null; is set by UWikiFormat if called as part of formatting chain.
  protected $nextID = 0;  // access via GiveID().
  public $objectIDs = array();

  public $all = array();  // array( 'type' => array('treePos' => element, 'treePos' => ...), ... )
  public $headings;       // array( headingElement, ... )
  public $anchors;        // instance of UWikiAnchorList.
  protected $linkedTo;

  public $cachedReplacements;  // used by TextRemplacements() method.
  // used by 'wacko' on render time:
  public $citationLevel;  // used by Uwacko_Citation.
  // used by 'wacko' on parse time:
  public $footnoteReferrers, $followingFootnotes, $blockFootnotes;
  public $nextFootnoteAnchorIndex;

  // replace* - array( 'className' => array('regexp piece' => array( [arg, arg...] ), 'piece' => ...]) )
  public $text = array('replaceWord' => array(), 'replaceWithTrailPunct' => array());
  // list symbols here without spaces; you can use dashes (-) and escape sequences (\x0000).
  public $wordCharacters = 'a-zA-Zа-яА-ЯёЁ';
  // Entity list: http://www.w3schools.com/tags/ref_entities.asp
  public $entities = array();
  // default class for elements not mentioned here will be "default".
  public $styleAliases = array(), $elementTokens = array();
  public $interwiki = array(), $numberlinks = array();

  // Max look ahead to find where ((compound:// ends)). If no delimeter found within this substr
  // the whole tag will be considered without tokens. Compare results of parsing these 2 samples:
  // * ((looooooooooooooooooooooooooooooooooooooooooooooooooooooooong==//i)) => <a href=...
  // * ((abc==//i)) => is left as is because // was opened and not closed and we were able to find the delim.
  public $firstTokenPartLookAheadLength = 250;
  // Inclusive; affects both ending inside (()) and trailing (new) ending. E.g.: See the ((standard+s)) will work for
  // this >= 1. Look ((the+y))re will work if it's >= 2, etc. Set to 0 to disable ending change feature.
  public $longestLinkEnding = 4;
  // Fast formatting makes much less thorough process of text nodes which may result in broken HTML.
  // It also slightly changes a few things (e.g. terms get replaced in the whole document
  // regardless of where they were first defined). For example, enable this setting and try
  // formatting this (might be rare to encounter something like this but disabling fast
  // formatting makes it bullet-proof):   (?term second?) (?second desc?) term
  // Enabling this option may boost up performance by ~30% (memory usage might decrease by ~5%).
  // It heavily depends on number of various text replacements you have (nowrap, typographic,
  // acronyms, etc.) - large number of them will sensitively impact performance if fast processing is off.
  public $fastTextFormatting = false;
  public $fetchRemoteTitles = true;
  public $fetchRemoteTitlesMaxSize = 4096;

  public $classAliases = array();
  public $globalHtmlClasses = array();    // these classes will be added to most HTML tags being output.
  public $anchorize = array('wacko_Heading' => true, 'wacko_Paragraph' => true);
  // instead of using full element name another (shorter) prefix can be used: wacko_Paragraph_1.3-1 => p_1.3-1.
  // Can be empty ('') - in this case "prefix_" will not be prepended. Don't specify identical
  // prefixes for several classes. Underscore (_) is appended automatically.
  public $autoAnchorContractions = array('wacko_Paragraph' => '');
  // If true, will generate names for $anchorize elements like "1.2.1-5" where "1.2.1" means
  // that that element is located under a 1st ==3rd level heading==, 2nd ==2nd level== and
  // 1st ==1st level==. If false, preceding heading's anchor will be used instead: "3rd_level-5".
  // In any case section prefix ("...-") won't be shown if the doc has no headings or an element
  // is located before the first. Generally, numeric sections are less prone to doc source
  // editing since they tolerate changes in heading captions.
  public $autoAnchorNumericSection = false;
  // if true, ((#anchor)) (with neither == nor " ") defines an inline anchor in-pace;
  // if false, it links to anchor "anchor" on the current page instead.
  // ((#anchor==anchor)), ((#anchor anchor)) and ((#anchor==)) is a link regardless of this setting.
  public $inlineAnchors = true;
  public $maxStyleNameLength = 30;
  public $maxLineQuoteLevel = 5;  // line quotes with more ">"s will be ignored.
  public $extLinksTarget = '_blank';
  public $expandTerms = true;
  public $minTermLengthToExpand = 2;    // inclusive; later occurences of (?terms?) shorter than this won't be expanded.
  // * 'block' - treat inline footnotes exactly as block ones, just with a different syntax.
  // * 'tooltip' - foot [[*note]] => foot <span title="note">[?]</span>
  // * 'expanded' - expands them into the text: foot [[*note]] => foot <span>note</span>
  public $inlineFootnotesAs = 'tooltip';
  // true - will put all block footnotes at the end of document; false - leave where they are defined.
  public $footnotesAtTheEnd = false;
  public $unindentBlocks = false;
  public $enableHTML5 = false;         // if on, HTML output will use modern HTML 5 tags and tricks.
  public $quotes = array(array('«', '»'), array('“', '”'), array('‘', '’'));

  // Normally docs contain 1 ==heading== that sets its title and actually used headings are
  // ===Level 2=== thru =======Lv7=======. This is 'normal' mode. Others:
  // * 'shifted' - removes one "=" necessary to create a heading but doc title can no more
  //   be specified by a heading as there's no Lv1 anymore: ==Lv2== thru ======Lv. 6======.
  //   "==" is level 2 thus only available levels are h2-h6.
  // * 'shifted-N' - as 'shifted' but shifts headings by specified heading levels:
  //                 'shifted-2' equals 'shifted', 'shifted-3' makes ==Lv3== and so on.
  //                 Don't set to 'shifted-1' or "0", it might cause glitches - set to 'normal' instead.
  // * 'extended' - identical to 'shiften' but treats first (only) ==heading== as doc title
  //    while others ==headings== are treated as of level 2 (===Lv2=== of 'normal') - thus
  //    no need to use {{title}} or other - you can specify doc title via first ==heading==.
  // 'shifted' mode will be enabled automatically for 'normal' if document contains {{Title}}.
  public $headingMode = 'normal';
  // usually applications using UWiki output doc title elsewhere on their own.
  public $hideDocTitle = false;
  // 'prepend' all attachment's data before the rendered doc; 'normal' - do nothing;
  // an instance of UWikiDocument - mergess attachments into that doc overwriting existing ones.
  // Note that if it's not 'prepend' you need to collect attachment data (e.g. CSS and
  // JavaScript code) and output it on your own in a corresponding place.
  public $attachments = 'prepend';

  public $dubiousAsComments = true;   // if true, ?? also won't accept any styles (obviously) and will become multiline.
  public $showComments = true;
  public $noLineBreaksInsideParagraph = false;  // if true, \n are ignored; --- can be used to force a line break.
  // Example (1st, 3, 4 lines are separate; 2nd is the continuation of 1st):
  //   Once upon a time... We called it...
  // So we lived. Until...
  //   Then it came. The wall of fire...
  //   And then...
  // Indentation is at least 2 leading spaces. The setting will work for indented paragraphs as well.
  public $moreIndentBreaksParaLine = false;   // is ignored if $noLineBreaksInsideParagraph == false.
  // ^ when both noLineBreaksInsideParagraph & moreIndentBreaksParaLine are true paragraphs
  //   having 1 line and whitespace sufficient for indentation are never indented (otherwise
  //   normal first line indentation would cause indentation of whole para as there's just 1 line).
  // it's possible to place list item on several lines (by proper unindentation); this setting
  // make each new line in a list item be broken into separate line - similar to $noLineBreaksInsideParagraph
  // except the latter works for paragraph lines.
  public $breakListsOnNewline = false;
  public $imageSizeSeparators = 'xX*:×';  // case-sensitive; used in {{Image pic.jpg, Title, 100x100}}.

  public $wackoWikiHighlDir = 'wacko/highlight';
  public $linkExt = '.html';           // will be added to local links; can be empty.
  public $anchorPrefix = '';           // will be prepended to any anchor defined in the document.

  /* NOTE:
    If you're outputting your page in UTF-8 make sure all URLs below are also in
    UTF-8 - and remember that PHP's file system functions usually (albeit not
    always) return strings in an ASCII charset. Sometimes in UTF-8 though (OS-dependent)... */
  public $rootURL = '/';    // URL of doc/forum/etc. where generated pages are hosted; including trailing /
  // $rootURL must NOT be contained in $baseURL ($baseURL can be '');
  // it includes trailing /; $baseURL is prepended to image URLs, local links, etc.
  protected $baseURL = '';  // set via BaseURL().
  // for ((#self-anchor links)) this will be prefixed (useful when using <base href />
  // because there links with no page will refer to base URL isntead of current page).
  public $pageForSelfAnchorLinks = '';  // baseURL will NOT be prepended to self anchor links.
  public $mediaURL = '/uversewiki/media/';    // URL to media/ folder including trailing /
  public $smileyURL = '/uversewiki/media/smilies/';   // including trailing /; may differ from $mediaURL.

  function DraftMode($turnOn) {
      $turnOn = (bool) $turnOn;
    $this->showComments = $turnOn;
  }

  function NovelMode($turnOn) {
      $turnOn = (bool) $turnOn;
    $this->noLineBreaksInsideParagraph = $turnOn;
    $this->moreIndentBreaksParaLine = $turnOn;
  }

  function __construct() {
    $this->handlers = new UWikiHandlerList;
    $this->anchors = new UWikiAnchorList($this);
    $this->pager = new UWikiPagerStub;
    $this->wackoWikiHighlDir and $this->wackoWikiHighlDir = UWikiRootPath.'/'.$this->wackoWikiHighlDir;
  }

  function SerializeTo(UWikiSerializer $ser) {
    $ser->WriteBool( isset($this->linkedTo) );
    $this->linkedTo and $ser->WriteSettings($this->linkedTo);

    $ser->WriteInstance($this->format, array('$', 'SerializeTo'));
    $ser->WriteArrayUsing(array($ser, 'WriteElement'), $this->objectIDs);
    $this->linkedTo or $ser->WriteNestedHash($this->all, array($ser, 'WriteElement'));
    $ser->WriteArrayUsing(array($ser, 'WriteElement'), $this->headings);
    $ser->WriteInstance($this->anchors, array('$', 'SerializeTo'));
  }

    function UnserializeFrom(UWikiUnserializer $ser, &$newObj) {
      $newObj = $this;
      $this->Reset(true);

      $isLinked = $ser->ReadBool();
      if ($isLinked) {
        $this->linkedTo = $ser->ReadSettings();
        $this->LinkTo($this->linkedTo);
      }

      $this->format = $ser->ReadInstanceOf('UWikiFormat', array('UWikiFormat', 'UnserializeFrom'));
      $this->linkedTo or $this->all = $ser->ReadNestedHash( array($ser, 'ReadElement') );
      $this->headings = $ser->ReadArrayUsing( array($ser, 'ReadElement') );
      $this->anchors = $ser->ReadInstanceOf('UWikiAnchorList', array('UWikiAnchorList', 'UnserializeFrom', $this));
    }

  function __clone() {
    // not cloning pager and others because it's a shared resource;
    // in contrast, handlers are part of environment that UWikiSettings represents.
    $this->handlers = clone $this->handlers;
  }

    function FullClone($which = array()) {
      $which += array('anchors' => true, 'pager' => true, 'format' => true);

      $clone = clone $this;

        empty($which['pager']) or $clone->pager = clone $this->pager;

        if (!empty($which['anchors'])) {
          $clone->anchors = clone $this->anchors;
          $clone->anchors->SetSettings($clone);
        }

        if (!empty($which['format']) and $this->format) {
          $clone->format = clone $this->format;
        }

      return $clone;
    }

    function LinkedClone() {
      $clone = clone $this;
      $clone->LinkTo($this);
      return $clone;
    }

    function LinkTo($master) {
      $this->linkedTo = $master;

      $this->all = &$master->all;
      $this->nextFootnoteAnchorIndex = &$master->nextFootnoteAnchorIndex;
      $this->nextID = &$master->nextID;
      $this->objectIDs = &$master->objectIDs;
      $this->attachments = &$master->attachments;
    }

  function Reset($full = true) {
    $this->ClearTextReplCache();

    $this->footnoteReferrers = array();
    $this->followingFootnotes = array();
    $this->headings = array();
    $this->blockFootnotes = array();
    $this->linkedTo = null;

    if ($full) {
      $this->citationLevel = -1;
      $this->nextFootnoteAnchorIndex = 1;
      $this->nextID = 0;
      $this->objectIDs = array();
      $this->anchors->Clear();
      $this->all = array('headings' => array(), 'blockFootnotes' => array());
    }
  }

  function ClearTextReplCache() { $this->cachedReplacements = null; }


  /* Utility methods: */

  static function LoadKeyValuesFrom($file, $separator = '\s+', $unescapeKey = false) {
    $result = array();

    if (is_file($file)) {
      foreach (file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) as $line) {
        if (!strpbrk($line[0], self::CommentPrefixes) and ($line = trim($line)) !== '') {
          @list($key, $value) = preg_split("/$separator/u", $line, 2);

          if (isset($value)) {
            $key = trim($key);
            $unescapeKey and $key = stripcslashes($key);
            $result[$key] = stripcslashes( ltrim($value) );
          }
        }
      }
    }

    return $result;
  }

    static function LoadArrayFrom($file, $valueKeys, $unescapeKey = false) {
      $result = array();

      if (is_file($file)) {
        $valueKeyCount = count($valueKeys);
        foreach (file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) as $line) {
          if (!strpbrk($line[0], self::CommentPrefixes) and ($line = trim($line)) !== '') {
            $values = preg_split('/\s+/u', $line, $valueKeyCount + 1);
            if (isset( $values[1] )) {   // at least a key and 1 value is mandatory for any item.
              foreach ($values as $i => &$value) {
                if ($i > 0 or $unescapeKey) {
                  $value = stripcslashes($value);
                }
              }

              $values = array_pad($values, $valueKeyCount + 1, '');
              $result[ array_shift($values) ] = array_combine($valueKeys, $values);
            }
          }
        }
      }

      return $result;
    }

  function FloodReplaceTerm($term, $meaningOrElement) {
    $term = preg_quote($term, '/');
    $this->text['replaceWord']['UWikiReplacedTerm'][$term] = array($meaningOrElement);

    // cache will be ignored if counts of replaces differ but if term already existed but
    // was changed it'll be used - wrongly. E.g: "(?term desc?) (?term second?) term"
    $this->ClearTextReplCache();
  }

  // Load settings after props like wordCharacters since regexps that are created use them.
  // $which is a list of files (w/o ext); if prefixed with 'all but' - loads all but listed;
  // otherwise loads listed only. E.g: 'all', '', 'all but terms', 'interwiki numberlinks extra'.
  function LoadFrom($path, $which = 'all') {
      $path = rtrim($path, '\\/');
      $allBut = substr("$which ", 0, 3) === 'all' ? false : true;

    if (strpos($which, 'terms') == $allBut)
      $this->LoadTermsFrom("$path/terms.conf");
    if (strpos($which, 'nowrap') == $allBut)
      $this->LoadNoWrapFrom("$path/nowrap.conf");

    if (strpos($which, 'smilies') == $allBut)
      $this->LoadReplacementsFrom("$path/smilies.conf");
    if (strpos($which, 'replace') == $allBut)
      $this->LoadReplacementsFrom("$path/replace.conf");

    if (strpos($which, 'style_aliases') == $allBut)
      $this->LoadStyleAliasesFrom("$path/style_aliases.conf");
    if (strpos($which, 'entities') == $allBut)
      $this->LoadEntitiesFrom("$path/entities.conf");
    if (strpos($which, 'strings') == $allBut)
      $this->LoadStringsFrom("$path/strings.conf");
    if (strpos($which, 'interwiki') == $allBut)
      $this->LoadInterwikiFrom("$path/interwiki.conf");
    if (strpos($which, 'numberlinks') == $allBut)
      $this->LoadNumberwikiFrom("$path/numberlinks.conf");
    if (strpos($which, 'element_tokens') == $allBut)
      $this->LoadElementTokensFrom("$path/element_tokens.conf");

    if (strpos($which, 'extra') == $allBut)
      $this->LoadExtra("$path/extra.php");
  }

    protected function LoadExtra($_file) { is_file($_file) and (include $_file); }

    // These methods exist for individual config files format of which might be changed later.
    function LoadStyleAliasesFrom($file) {
      $this->styleAliases = self::LoadKeyValuesFrom($file) + $this->styleAliases;
    }

    function LoadEntitiesFrom($file) {
      $this->entities = self::LoadArrayFrom($file, array('code', 'symbol')) + $this->entities;
    }

    function LoadStringsFrom($file) {
      $this->strings = self::LoadKeyValuesFrom($file, '=') + $this->strings;
    }

    function LoadInterwikiFrom($file) {
      $this->interwiki = self::LoadArrayFrom($file, array('target', 'root'), true) + $this->interwiki;
    }

    function LoadNumberwikiFrom($file) {
      $this->numberlinks = self::LoadArrayFrom($file, array('target', 'root'), true) + $this->numberlinks;
    }

    function LoadElementTokensFrom($file) {
      $tokens = self::LoadKeyValuesFrom($file);
        foreach ($tokens as &$token) {
          $token = explode(',', $token);
          foreach ($token as &$elemName) { $elemName = trim($elemName); }
        }

      $this->elementTokens = $tokens + $this->elementTokens;
    }

    function LoadNoWrapFrom($file) {
      if (is_file($file)) {
        foreach (file($file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES) as $line) {
          if (!strpbrk($line[0], self::CommentPrefixes) and ($line = trim($line)) !== '') {
            $substr = str_replace( '\\\d', '\d+', preg_quote($line, '/') );
            $this->text['replaceWithTrailPunct']['UWikiReplacedNoWrap'][$substr] = array();

            $this->ClearTextReplCache();
          }
        }
      }
    }

    function LoadTermsFrom($file) {
      foreach (self::LoadKeyValuesFrom($file) as $term => $meaning) {
        $this->FloodReplaceTerm($term, $meaning);
      }
    }

    // arbitrary number of arguments allowed.
    function LoadReplacementsFrom($file_1) {
      $list = array();
      foreach (func_get_args() as $file) { $list += self::LoadArrayFrom($file, array('image', 'text')); }

      $htmlPieces = $textPieces = '';
      foreach ($list as $substr => $item) {
        $substr = preg_quote($substr, '/');
        $this->text['replaceWord']['UWikiReplacement'][$substr] = $item;
      }

      $this->ClearTextReplCache();
    }

  // called by UWikiFormattedText.
  function TextReplacements() {
    $result = array();

    if ($this->cachedReplacements) {
      $result = $this->cachedReplacements;
    } else {
      list($allPieces, $classes) = self::ReplacementFrom( $this->text['replaceWord'] );
      if (!empty($classes)) {
        $pattern = '/(?<='.UWikiLeftWordBoundary.")($allPieces)(?=".UWikiRightWordBoundary.')/u';
        $result[$pattern] = $classes;
      }

      list($allPieces, $classes) = self::ReplacementFrom( $this->text['replaceWithTrailPunct'] );
      if (!empty($classes)) {
        $pattern = '/(?<='.UWikiLeftWordBoundary.")($allPieces)(".UWikiRightWordBoundary.')/u';
        $result[$pattern] = $classes;
      }

      $this->cachedReplacements = $result;
    }

    $this->handlers->Call('text replace', array(&$result, $this));
    return $result;
  }

    // $list is an array of { 'className' => array('regexp' => array( [arg, arg, ...] )) }
    static function ReplacementFrom($list) {
      $classes = array();   // array( array('className', [arg, ...]), array('class', ...) )
      $allPieces = '';

      foreach ($list as $class => $regexps) {
        if (!empty($regexps)) {
          $allPieces .= '('. join( ')|(', array_keys($regexps) ) .')|';

          foreach ($regexps as &$info) {
            array_unshift($info, $class);
            $classes[] = $info;
          }
        }
      }

      return array(substr($allPieces, 0, -1), $classes);
    }

  // $newCurrentPage is relative to current $baseURL unless it's absolute (see PathType()).
  function AlterPathsBy($newCurrentPage) {
    if (UWikiBaseElement::IsPath('rel', $newCurrentPage)) {
      $base = rtrim($this->BaseURL(), '\\/').'/';
    } else {
      $base = '';
    }

      if (strpbrk($newCurrentPage, '\\/')) {
        $this->BaseURL( $base.ltrim(dirname($newCurrentPage), '\\/') );
      }

    $this->pager->SetCurrent($newCurrentPage);
  }

  // Base URL must not include root URL; may be ''. Always returns no leading but one trailing /.
  function BaseURL($newURL = null) {
    if ($newURL !== null) {
      $newURL = trim($newURL, '\\/');
      $this->baseURL = $newURL === '' ? '' : "$newURL/";
    }

    return $this->baseURL;
  }

    function BaseUrlWithRoot($newURL = null) {
      if ($newURL !== null) {
        if (strpos($newURL, $this->rootURL) !== 0) {
          throw new EUverseWiki(__CLASS__.'->BaseUrlWithRoot() failed because $newURL didn\'t contain current root URL.');
        } else {
          $this->BaseURL( substr($newURL, strlen($this->rootURL)) );
        }
      }

      return $this->rootURL.$this->BaseURL();
    }

  function GiveID() { return ++$this->nextID; }

    function GiveIdTo($obj) {
      $id = ++$this->nextID;
      $this->objectIDs[$id] = $obj;
      return $id;
    }

    function ObjectByID($id) { return $this->objectIDs[$id]; }
    function GetObjectIDs() { return array_keys($this->objectIDs); }

    function ElementByID($id) {
      $obj = $this->objectIDs[$id];
      if ($obj and $obj instanceof UWikiBaseElement) { return $obj; }
    }
}

  class UWikiHandlerList {
    // an array of groups which is either an array of callbacks or array('token', 'Class', flags).
    protected $handlers = array();

    function HasAny($group) { return (bool) $this->GetRef($group); }
    function Call($group, $args = array()) { UWikiDocument::CallAll($this->Get($group), $args); }

    function Set($group, $handlers) {
      $handlersRef = &$this->GetRef($group);
      $handlersRef = $handlers;
    }

      function SetAllFrom($groups) {
        $this->Clear();

        if ($groups) {
          foreach ($groups as $group => $toSetTo) {
            $handlers = &$this->GetRef($group);
            $handlers = $toSetTo;
          }
        }
      }

    function Get($group) {
      $handlers = &$this->GetRef($group);
      return $handlers ? $handlers : array();
    }

      protected function &GetRef($group) { return $this->handlers[strtolower($group)]; }

    function AddTo($group, $handler) {
      $handlers = &$this->GetRef($group);
      $handlers[] = $handler;
      return count($handler) - 1;
    }

    function AddOrReplace($key, $group, $handler) {
      $handlers = &$this->GetRef($group);
      $handlers[$key] = $handler;
    }

    function Clear($group = null) {
      if ($group === null) {
        $this->handlers = array();
      } else {
        $handlers = &$this->GetRef($group);
        $handlers = array();
      }
    }

    function RemoveOrThrow($group, $keyOrHandler) {
      if (!$this->RemoveIfExists($group, $keyOrHandler)) {
        if (is_array($keyOrHandler)) {
          foreach ($keyOrHandler as &$item) {
            if (is_scalar($item)) {
              $item = var_export($item, true);  // var_export() fails on too deep recursion, e.g. object's.
            } elseif (is_object($item)) {
              $item = '<'.get_class($item).'>';
            } else {
              $item = gettype($item);
            }
          }

          $keyOrHandler = 'array('. join(', ', $keyOrHandler) .')';
        }

        throw new EUverseWiki("Cannot remove handler $keyOrHandler from group $group.");
      }
    }

      function RemoveIfExists($group, $keyOrHandler) {
        $handlers = &$this->GetRef($group);

        if ($handlers) {
          if (is_scalar($keyOrHandler)) {
            if (isset( $handlers[$keyOrHandler] )) {
              unset( $handlers[$keyOrHandler] );
              return true;
            }
          } else {
            foreach ($handlers as $i => &$handler) {
              if ($handler === $keyOrHandler) {
                unset( $handlers[$i] );
                return true;
              }
            }
          }
        }
      }
  }

  class UWikiAnchorList {
    protected $settings, $strings;

    protected $elementsByAnchors;   // 'anchorName' => UWikiBaseElement
    protected $counts;              // 'elementName' => count
    protected $section;             // e.g. array(1, 2, 3) for '1.2.3. Heading'
    protected $sectionAnchorName;   // used when $autoAnchorNumericSection is false.

    function __construct($settings) {
      $this->SetSettings($settings);
      $this->Clear();
    }

      function SetSettings($settings) {
        $this->settings = $settings;
        $this->strings = &$this->settings->strings;
      }

      function Clear() {
        $this->elementsByAnchors = array();
        $this->NewSection(array(), '');
      }

    function SerializeTo(UWikiSerializer $ser) {
      $anchors = array_keys($this->elementsByAnchors);
      $ser->WriteArray($anchors);
      $anchors and $ser->WriteArrayUsing(array($ser, 'WriteElement'), $this->elementsByAnchors);

      $ser->WriteArray($this->counts);
      $ser->WriteArray($this->section);
    }

      static function UnserializeFrom(UWikiUnserializer $ser, &$obj, $class, UWikiSettings $settings) {
        if (!class_exists($class)) {
          $ser->Error('settings\' anchor list class '.$class.' is undefined');
        }

        $obj = new $class($settings);

        $anchors = $ser->ReadArray();
        $anchors and $elements = $ser->ReadArrayUsing( array($ser, 'ReadElement') );
        $obj->elementsByAnchors = $anchors ? array_combine($anchors, $elements) : array();

        $obj->counts = $ser->ReadArray();
        $obj->section = $ser->ReadArray();
      }

    function CountOf($elementName) {
      $count = &$this->counts[$elementName];
      return $count ? $count : 0;
    }

      protected function CountAndInc($elementName) {
        return ($this->counts[$elementName] = $this->CountOf($elementName) + 1);
      }

    function CurrentSection() {
      return $this->section ? $this->section : array(0);
    }

      function CurrentSectionAnchorName() { return $this->sectionAnchorName; }

      function NewSection($section, $anchorName) {
        $this->section = $section;
        $this->sectionAnchorName = $anchorName;
        $this->ClearCounts();
      }

        protected function ClearCounts() { $this->counts = array(); }

    function GenerateFor($element) {
      return $element->kind === 'heading' ? $this->GenerateHeadingNameFor($element)
                                          : $this->GenerateRegularNameFor($element);
    }

      protected function GenerateRegularNameFor($element) {
        $elementName = $element->elementName;

        $count = $this->CountAndInc($elementName);

        $contracted = &$this->settings->autoAnchorContractions[$elementName];
        if ($contracted === '') {
          $prefix = '';
        } else {
          $prefix = UWikiBaseElement::IsEmptyStr($contracted) ? $elementName : $contracted;
          $prefix = sprintf($this->strings['anchor name prefix'], $prefix);
        }

        if ($this->settings->autoAnchorNumericSection) {
          $section = $this->CurrentSection();
          $section = join($this->strings['anchor name section delimiter'], $section);
          $section === '0' and $section = '';
        } else {
          $section = $this->CurrentSectionAnchorName();
        }

        $section === '' or $count = sprintf($this->strings['anchor name'], $section, $count);
        $name = $prefix.$count;

        $name = $this->UniqueNameFrom($name, $element);
        return $name;
      }

      protected function GenerateHeadingNameFor($heading) {
        // todo: use ToText instead of strip_tags when it's implemented.
        $rendered = ($heading->doc->IsBeingParsed() or $heading->doc->IsRenderedInto('html'));
        $rendered or $heading->doc->BeginRenderingInto('html');
        $text = strip_tags($heading->ChildrenToHTML());
        $rendered or $heading->doc->EndRenderingInto('html', $text);

        $name = $this->UniqueNameFrom($text, $heading);

        $level = $heading->level;
        $section = array_pad( array_slice($this->CurrentSection(), 0, $level), $level, 0 );
        ++$section[$level - 1];
        $this->NewSection($section, $name);

        return $name;
      }

    function GenerateAndRegisterFor($element) {
      $name = $this->GenerateFor($element);
      $this->Register($name, $element);
      return $name;
    }

    function IsFree($name)  { return  empty( $this->elementsByAnchors[$name] ); }
    function IsTaken($name) { return !empty( $this->elementsByAnchors[$name] ); }

    function ElementBy($name) {
      $element = &$this->elementsByAnchors[$name];
      return is_object($element) ? $element : null;
    }

      function HeadingBy($name) {
        $element = $this->ElementBy($name);
        return (($element and $element->kind === 'heading') ? $element : null);
      }

      function ElementBySimilarAnchor($strOrArray) {
        $pieces = $this->Split($strOrArray);

          $name = '';
          do {
            $name === '' or $name .= '_';
            $name .= array_shift($pieces);
            $obj = $this->HeadingBy($name);
          } while (!is_object($obj) and $pieces);

        return $obj ? $name : null;
      }

    function UniqueNameFrom($strOrArray, $forElement = null) {
      $pieces = $this->Split($strOrArray);

        $name = '';
        do {
          do {
            $piece = array_shift($pieces);
            if ($piece === null) {
              if ($this->IsTaken($name)) {
                $suffix = 0;
                while ($this->IsTaken( $name.'_'.++$suffix )) { }
                $name .= "_$suffix";
              }

              break;
            } else {
              $name === '' or $name .= '_';
              $name .= $piece;
            }
          } while (!isset( $piece[UWikiMinAnchorPieceLength - 1] ));

          $heading = $this->HeadingBy($name);
          if ($heading and !$heading->retainAnchor and $heading !== $forElement) {
            $heading->Anchorize();
            $heading->retainAnchor = true;
            // Note 1: we could unset the name now since it's unused but we don't so future anchors won't
            // repeat this ambiguous name. E.g.: ==name 1==\n ==name 2==\n ==name 3== would have
            // anchors name_1, name_2 and name; w/o unsetting the name here we get name_1, name_2 and name_3.
            // Note 2: we set $retainAnchor of the heading to false so it won't be re-anchorized. Example:
            // ==name x==\n ==name==\n ==name== => name_x, name_1, name_2 (when setting to null). And if
            // we left it "name x" would be re-anchorized on ALL following "name"s, resulting in
            // name_x_1, name_1, name_2. Add another ==name== and you'll get name_x_2, etc.
            // Note 3: there's still a glitch with this approach, e.g.: ==name x==\n ==name x==\n
            // ==name==\n ==name==\n => anchors name_x, name_x_1, name_1, name_2 - but one can't seriously
            // expect great efficiency of auto-anchorizing headings with the same contents anyway.
          }
        } while ($this->IsTaken($name));

      return $name;
    }

      function Split($strOrArray) {
        if (is_array($strOrArray)) {
          return $strOrArray;
        } else {
          $wordCh = $this->settings->wordCharacters;
          return preg_split('/[^'.$wordCh.'0-9._-]+/iu', mb_strtolower($strOrArray),
                            -1, PREG_SPLIT_NO_EMPTY);
        }
      }

    function Register($anchorName, $element) {
      if ($this->IsTaken($anchorName)) {
        throw new EUverseWiki("Cannot register anchor named $anchorName because this name is already taken.");
      } else {
        $this->ReRegister($anchorName, $element);
      }
    }

      function ReRegister($anchorName, $element) {
        $this->elementsByAnchors[$anchorName] = $element;
      }
  }
