<?php
/*
  This action generates Table of Contents for the current document.
  Outputs nothing if tehre were no headings with level > 1.

    {{TOC up to=3}} or just {{TOC 3}}   - include headings with levels below and
                                          including 3rd (no 4th and below: ====skipped====)

  By default TOC is generated for current document that contains the {{TOC}} string.
  You can generate TOC for topmost doc (most outer) by chaining TOC with TopDoc:
    {{TopDoc; TOC}}
*/

class Utoc_Root extends UWikiBaseAction {
  public $isBlock = true;
  public $htmlTag = 'fieldset';
  public $htmlClasses = array('toc');
  public $parameterOrder = array('up to');

  function Execute($format, $params) {
    $format->appendMe = true;
    $items = array();

    $upTo = &$params['up to'];
    $upTo = is_numeric($upTo) ? max($upTo, 2) : 6;

      foreach ($format->settings->all['headings'] as $obj) {
        if ($obj->level > 1 and $obj->level <= $upTo) {
          $link = $items[] = $this->NewElement('Utoc_Heading');
          $link->LinkTo($obj);
        }
      }
      isset($obj) and $format->origDoc->settings->anchorize[$obj->elementName] = true;

    if ($items) {
      if ($title = $format->origDoc->GetTitle()) {
        $titleObj = $this->children[] = $this->NewElement('Utoc_DocTItle');
        $titleObj->heading = $title;
      }

        if (strpos($this->settings->headingMode, 'shifted-') === 0) {
          $rootLevel = ((int) substr($this->settings->headingMode, -1));
        } else {
          $rootLevel = 2;
        }

      $this->children[] = $this->BuildListFrom($items, 'Utoc_MissingHeading',
                                               'Utoc_Headings', $rootLevel);
    } else {
      $this->htmlTag = null;
    }
  }

    function PrepareItem($item, &$stack) {
      if ($item->level !== 1) {
        $index = $item->NewElement('Utoc_SectionIndex');

          foreach ($stack as $i => $obj) {
            array_unshift($index->index, count($obj->children));
            $i === 0 and ++$index->index[0];
          }

        array_unshift($item->children, $index);
      }
    }

  function IsDynamic($format, $params) { return true; }
  function Priority($format, $params) { return (int) UWikiMaxPriority * 0.3; }
}

  class Utoc_DocTItle extends UWikiBaseElement {
    public $htmlTag = 'legend';
    public $isBlock = true;
    public $heading;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);
      $props['element'][] = 'heading';
    }

    function SelfToHtmlWith($contents) {
      $title = $this->heading;

      if (is_object($title)) {
        $rendered = ($title->doc->IsBeingParsed() or $title->doc->IsRenderedInto('html'));
        $rendered or $title->doc->BeginRenderingInto('html');
        $html = $title->ChildrenToHTML(false);
        $rendered or $title->doc->EndRenderingInto('html', $html);
      } else {
        $html = $title;
      }

      return parent::SelfToHtmlWith($html);
    }
  }

  class Utoc_Headings extends UWikiBaseElement {
    public $htmlTag = 'ol';
    public $isBlock = true;
    public $level;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);
      $props['int'][] = 'level';
    }

    function Group($childItem) { }
  }

    class Utoc_Heading extends UWikiBaseElement {
      public $htmlTag = 'li';
      public $isBlock = true;
      public $heading, $level;

      function SetupSerialize(array &$props) {
        parent::SetupSerialize($props);

        $props['element'][] = 'heading';
        $props['int'][] = 'level';
      }

      function LinkTo($headingElement) {
        $link = $this->children[] = $this->NewElement('Utoc_Link');
        $this->heading = $link->heading = $headingElement;
        $this->level = $headingElement->level;
      }
    }

      class Utoc_Link extends Utoc_DocTItle {
        public $htmlTag = 'a';
        public $isSingleLineHTML = true;  // browsers output newlines as spaces inside link.

        function AllToHTML() { return ' '.parent::AllToHTML(); }

        function SelfHtmlAttributes() {
          $attrs = array();
          if ($this->heading->anchor) {
            $attrs['href'] = $this->settings->pageForSelfAnchorLinks.
                             '#'.$this->settings->anchorPrefix.$this->heading->anchor->Name();
          }
          return $attrs;
        }
      }

    class Utoc_MissingHeading extends UWikiBaseElement {
      public $htmlTag = 'li';
      public $isBlock = true;
      public $level;

      function SetupSerialize(array &$props) {
        parent::SetupSerialize($props);
        $props['int'][] = 'level';
      }

      function Pad($item) {
        $this->children[] = $this->NewElement('Utoc_MissingHeadingText');
      }
    }

      class Utoc_MissingHeadingText extends UWikiFormatErrorMessage {
        public $htmlClasses = array('missing-heading');
        public $message = '{{toc: missing heading';
        public $defaultMessage = 'Missing heading.';
      }

    class Utoc_SectionIndex extends UWikiBaseElement {
      public $htmlTag = 'span';
      public $htmlClasses = array('index');

      public $index = array();

      function SetupSerialize(array &$props) {
        parent::SetupSerialize($props);
        $props['list'][] = 'index';
      }

      function SelfToHtmlWith($contents) {
        $delim = @$this->strings['{{toc: section delimiter'];
        $delim or $delim = '.';
        $index = join($delim, $this->index);

        $msg = @$this->strings['{{toc: section'];
        $msg or $msg = '%s.';
        return parent::SelfToHtmlWith( sprintf($msg, $index) );
      }
    }
