<?php
  /* UverseWiki - a full-fledged modelling text processor; http://uverse.i-forge.net/wiki */
  /* Make sure all input strings are in UTF-8. */

  define('UverseWikiVersion', 0.9);
  define('UverseWikiBuild', (int) substr('$Rev: 507 $', 5, -1));
  define('UverseWikiHomePage', 'http://uverse.i-forge.net/wiki/');

  defined('UWikiRootPath') or define('UWikiRootPath', rtrim(dirname(__FILE__), '\\/'));

  define('UWikiMaxPriority', 10);  // inclusive; max priority of a markup (UWikiBaseELement->Priority).
  define('UWikiWikiMarkup', 'wacko');
  define('UWikiTextMarkup', 'text');
  define('UWikiInterwikiSeparator', ':');
  define('UWikiMinAnchorPieceLength', 3);   // inclusive.


class EUverseWiki extends Exception { }

  class EUWikiLastPCRE extends EUverseWiki {
    public $pcreErrorCode;

    static function ThrowIfPcreFailed() {
      if (preg_last_error() != PREG_NO_ERROR) { throw new self; }
    }

    function __construct() {
      $this->pcreErrorCode = preg_last_error();
      parent::__construct('PCRE error; preg_last_error() returned '.$this->pcreErrorCode.
                          '. Check the encoding of your document - it must be UTF-8.');
    }
  }

  abstract class EUWikiDocument extends EUverseWiki { }

    class EUWikiInvalidMarkupName extends EUWikiDocument {
      public $markup;

      function __construct($markup) {
        parent::__construct("Cannot load \"$markup\" markup - invalid name (only a-z, 0-9 are acceptable).");
      }
    }

    class EUWikiCannotLoadMarkup extends EUWikiDocument {
      public $markup;

      function __construct($markup) {
        parent::__construct("Cannot load \"$markup\" markup - its root class doesn't exist.");
      }
    }

    class EUWikiNoMarkupLoaded extends EUWikiDocument {
      function __construct() { parent::__construct('No markup has been loaded.'); }
    }

    class EUWikiRenderingNotStarted extends EUWikiDocument {
      public $markup;

      function __construct($markup) {
        $this->markup = $markup;
        parent::__construct('One of UWikiDocument elements attempted to render itself into'.
                            " $markup but BeginRenderingInto() was not called beforehand.");
      }
    }

    class EUWikiInliningBlocks extends EUWikiDocument {
      public $parent;

      function __construct(UWikiBaseElement $parent) {
        $this->parent = $parent;
        parent::__construct("Attempted to inline {$parent->className} but it only contained block children.");
      }
    }

  class EUWikiDeprecated extends EUverseWiki {
    public $feature, $issuer, $desc;

    function __construct($feature, $issuer = null, $desc = null) {
      $this->feature = $feature;
      $this->issuer = $issuer;
      $this->desc = $desc;

      $issuer and $feature .= ', issued by '.$issuer;
      parent::__construct('UverseWiki: deprecated feature - '.$feature.'. '.$desc);
    }
  }


class UWikiDocument {
  static $loadedHandlers = array();
  // all dirs that exist in UWikiRootPath but are not markups; *.php will never be included from them.
  static $ignoreMarkupDirNames = array('autoload', 'config', 'extras', 'tests', 'media');

  static $onCreateHooks = array();                            // function (UWikiDocument $doc)
  static $setSourceHooks = array();                           // function (&$source, UWikiDocument $doc)
    public $setSourceToThisHooks = array();
  static $beforeParsingHooks = array();                       // function (UWikiDocument $doc)
    public $beforeParsingThisHooks = array();
  static $afterParsingHooks = array();                        // function (UWikiDocument $doc)
    public $afterParsingThisHooks = array();
  static $beforeRenderingHooks = array();                     // function ($format, UWikiDocument $doc)
    public $beforeRenderingThisHooks = array();
  static $afterRenderingHooks = array();                      // function ($format, &$result, UWikiDocument $doc)
    public $afterRenderingThisHooks = array();
  public $newDocumentHooks = array();                         // function (UWikiDocument &$obj, UWikiDocument $owner)
  public $newElementHooks = array('any' => array());          // function (UWikiBaseElement &$obj)
  // function (UWikiBaseElement $element, array &$attrs, &$contents, &$htmlPrefix, &$htmlSuffix)
  public $onHtmlTagHooks = array('any' => array());

  // see ::Quote(). function (&$str, &$markup)
  static $quoteHooks = array('wacko' => array(__CLASS__, 'QuoteWacko'));
  // see ::GetTitleOf(). function (&$gotTitle, &$doc, &$markup, &$quick)
  static $getTitleHooks = array('wacko' => array(__CLASS__, 'GetTitleOfCore'));

  // "exception", "message" (via echo), false (ignore),
  // callable (call back but continue): function ($instance_of_EUWikiDeprecated)
  static $deprecatedReporting = 'exception';

  public $version, $build, $settings, $root, $style, $elements;
  // $title is either a string (plain text), an UWikiBaseELement or null. if doc has
  // a 1-st level heading it takes precedence - see GetTitle().
  public $title, $meta = array();

  // array of array; standard keys: css, js; might contain others (plugin-specific).
  public $attachments = array();

  protected $source;
  protected $id;        // access via ID().
  protected $isParsed = false;
  protected $renderedInto = array();

  static function Initialize() {
    include_once UWikiRootPath.'/autoload/base.php';
    foreach (glob(UWikiRootPath.'/autoload/*.php', GLOB_NOSORT) as $file) { include_once $file; }
  }

  static function RootElemNameFor($markup) { return "U{$markup}_Root"; }
  static function MarkupDirOf($markup) { return UWikiRootPath."/$markup"; }

    static function MarkupExists($markup) {
      return class_exists( self::RootElemNameFor($markup) )
             or (array_search(strtolower($markup), self::$ignoreMarkupDirNames) === false
                 and is_file( self::MarkupDirOf($markup).'/base.php' ));
    }

    static function ExistingMarkups($filterByProp = null) {
      $markups = array();

        foreach (glob(self::MarkupDirOf('*').'/base.php') as $dir) {
          $markups[ strtolower(ltrim( strrchr(dirname($dir), '/'), '/' )) ] = true;
        }

        foreach (get_declared_classes() as $class) {
          if ($class[0] === 'U' and substr($class, -5) === '_Root') {
            if ($filterByProp) {
              $root = new $class;
              if (!$root->$filterByProp) { continue; }
            }

            $markups[ substr($class, 1, -5) ] = true;
          }
        }

      foreach (self::$ignoreMarkupDirNames as $dir) { unset( $markups[$dir] ); }
      return array_keys($markups);
    }

      static function ExistingFormatters() {
        return self::ExistingMarkups('isFormatter');
      }

      static function ExistingActions() {
        return self::ExistingMarkups('isAction');
      }

    protected static function TestMarkupPropOf($markup, $prop, $valueIfSeparate) {
      if (is_file(self::MarkupDirOf($markup).'/base.php')) {
        return $valueIfSeparate;
      } else {
        $class = self::RootElemNameFor($markup);
        $root = new $class;
        return $root->$prop;
      }
    }

      static function IsFormatter($markup) {
        return self::TestMarkupPropOf($markup, 'isFormatter', true);
      }

      static function IsAction($markup) {
        return self::TestMarkupPropOf($markup, 'isAction', false);
      }

    function CreateRootElemOf($markup) {
      self::PreloadMarkup($markup);
      $obj = $this->NewElement( self::RootElemNameFor($markup) );
      $obj->source = &$this->source;
      return $obj;
    }

  static function PreloadMarkup($markup) {
    $markup = strtolower($markup);
    // '_' is not allowed because it's a separator in class anmes: Umarkup_Element
    if (ltrim($markup, 'a..z0..9') !== '') { throw new EUWikiInvalidMarkupName($markup); }
    $rootClass = self::RootElemNameFor($markup);

    if (!class_exists($rootClass) and self::MarkupExists($markup)) {
      $markupDir = self::MarkupDirOf($markup);
      if (is_dir($markupDir)) {
        is_file("$markupDir/base.php") and (include_once("$markupDir/base.php"));
        foreach (glob("$markupDir/*.php", GLOB_NOSORT) as $file) { self::LoadModuleInto($markup, $file); }
      }
    }

    if (!class_exists($rootClass)) { throw new EUWikiCannotLoadMarkup($markup); }
  }

    static function LoadModuleInto($markup, $_file) {
      // You can't just include() from arbitrary place in the code since modules access
      // static props/methods of UWikiDocument, referring to self.
      return include_once($_file);
    }

  // if $markup isn't passed it'll be set to $settings->markupName.
  static function TryParsing($source, $settings, $markup = null) {
    if (is_string($settings) and !$markup) {
      $markup = $settings;
      $settings = null;
    }

    $markup or $markup = $settings->markupName;

    if (self::MarkupExists($markup)) {
      try {
        $doc = new self($source);
        $settings ? ($doc->settings = $settings) : $doc->settings->LoadFrom(UWikiRootPath.'/config');
        $doc->LoadMarkup($markup);
        $doc->Parse();
        return $doc;
      } catch (Exception $e) {
      }
    }
  }

  function __construct($source) {
    $this->settings = new UWikiSettings;
    $this->style = new UWikiStyle($this->settings);

    self::CallAll(self::$onCreateHooks, array($this));
    $this->SetSource($source);
  }

    function SetSource($source) {
      if ($this->IsParsed() or !empty($this->renderedInto)) {
        throw new EUverseWiki('Cannot set new source to already parsed or rendered document.');
      }

      // A doc might have different $version/$build if we load (e.g. unserialize) an older one.
      $this->version = UverseWikiVersion;
      $this->build = UverseWikiBuild;

      if ($source === '') {
        $this->source = '';
      } else {
        self::CallAll($this->setSourceToThisHooks, array(&$source, $this));
        self::CallAll(self::$setSourceHooks, array(&$source, $this));

        $this->source = $this->CleanSource($source);
      }
    }

      static function &CleanSource(&$source) {
        // removes trailing whitespace before each newline.
        $result = preg_replace("/[ \t\v]+(\r?\n|$)/u", '\1', $source);
        EUWikiLastPCRE::ThrowIfPcreFailed();
        return $result;
      }

  function GetSource() { return $this->source; }
  function LoadedMarkup() { return $this->root ? $this->settings->markupName : null; }

  // returns null for system classes (not containing markup name, e.g. UWikiTextFromRaw).
  static function MarkupNameFrom($class) {
    if (strpos($class, '_') !== false) {
      $class[0] === 'U' and $class = substr($class, 1);
      return strtok($class, '_');
    }
  }

    static function ElementNameFrom($class) {
      strtok($class, '_');
      $name = strtok(null);
      return "$name" === '' ? $class : $name;
    }

  function LoadMarkup($markup) {
    $this->settings->markupName = $this->root = null;
    $this->root = $this->CreateRootElemOf($markup);

    $this->settings->markupName = strtolower($markup);
    $this->settings->handlers->SetAllFrom( @self::$loadedHandlers[$markup] );
  }

  function Parse() {
    if (!$this->LoadedMarkup()) { throw new EUWikiNoMarkupLoaded(); }
    if ($this->IsParsed()) { throw new UWikiDocument('Cannot parse the same UWikiDocument twice.'); }

    if (strtolower(mb_internal_encoding()) !== 'utf-8') {
      throw new EUverseWiki('mbstring\'s internal encoding must be set to UTF-8'.
                            ' before calling Parse().');
    }

    $this->elements = array();

    $fullReset = $this->settings->format == null;
    $this->settings->Reset($fullReset);
    $fullReset and $this->root->elementID = $this->settings->GiveIdTo($this->root);

    $root = $this->root;
    $root->children = array();
    $root->SetRaw($this->source);

    $root->BeforeParsing();
    $root->Parse();
    $root->AfterParsing();

    return $root;
  }

    function BeforeParsing() {
      $this->isParsed = 'being';

      self::CallAll($this->beforeParsingThisHooks, array($this));
      self::CallAll(self::$beforeParsingHooks, array($this));
    }

    function AfterParsing() {
      $this->isParsed = true;

      self::CallAll($this->afterParsingThisHooks, array($this));
      self::CallAll(self::$afterParsingHooks, array($this));
    }

  function RenderIn($format) {
    if (!$this->LoadedMarkup()) { throw new EUWikiNoMarkupLoaded(); }

    $format = strtolower($format);
    if ($format === 'html5') {
      $format = 'html';
      $oldHtml5Setting = $this->settings->enableHTML5;
      $this->settings->enableHTML5 = true;
    }

    $this->BeginRenderingInto($format);
      $renderFunc = 'AllTo'.strtoupper($format);
      $result = trim( $this->root->$renderFunc(), "\r\n" );
    $this->EndRenderingInto($format, $result);

    isset($oldHtml5Setting) and $this->settings->enableHTML5 = $oldHtml5Setting;
    return $result;
  }

  function IsRenderedInto($format) { return !empty($this->renderedInto[$format]); }
  function IsBeingParsed() { return $this->isParsed === 'being'; }
  function IsParsed() { return (bool) $this->isParsed; }

  function EnsureIsRenderedInto($format) {
    if (!$this->IsRenderedInto($format)) {
      throw new EUWikiRenderingNotStarted($format);
    }
  }

  function BeginRenderingInto($format) {
    if (!$this->IsRenderedInto($format)) {
      if ($this->IsBeingParsed()) {
        throw new EUverseWiki('Cannot begin rendering a document while it\'s being parsed.');
      } else {
        $this->renderedInto[$format] = 'being';
        $this->root->BeforeRenderingInto($format);
      }
    }
  }

  function EndRenderingInto($format, &$result) {
    $state = &$this->renderedInto[$format];
    if ($state === 'being') {
      $state = true;
      $this->root->AfterRenderingInto($format, $result);
    } elseif ($state !== true) {
      throw new EUverseWiki('EndRenderingInto() was called without BeginRenderingInto().');
    }
  }

    function BeforeRenderingInto($format) {
        $format = strtolower($format);
      self::CallAll($this->beforeRenderingThisHooks, array($format, $this));
      self::CallAll(self::$beforeRenderingHooks, array($format, $this));
    }

    function AfterRenderingInto($format, &$result) {
        $format = strtolower($format);
      self::CallAll($this->afterRenderingThisHooks, array($format, &$result, $this));
      self::CallAll(self::$afterRenderingHooks, array($format, &$result, $this));

      $this->HandleAttachmentsFor($format, $result);
    }

      function HandleAttachmentsFor($format, &$rendered) {
        static $html = array('css' => array('<style type="text/css">', '</style>'),
                             'js' => array('<script type="text/javascript">', '</script>'));

        $mode = $this->settings->attachments;
        if ($mode === 'normal') {
          // do nothing.
        } elseif ($mode == 'prepend') {
          $attachments = '';
          foreach ($this->attachments as $type => &$list) {
            if ($format === 'html' and $wrapper = &$html[$type]) {
              list($pf, $sf) = $wrapper;
            } else {
              $pf = $sf = '';
            }

            $attachments .= $pf.join("\n\n", $list).$sf;
          }

          $attachments === '' or $rendered = $attachments."\n\n".$rendered;
        } elseif ($mode instanceof self) {
          $merged = &$mode->attachments;

          foreach ($this->attachments as $type => &$list) {
            if (empty($merged[$type])) {
              $merged[$type] = $list;
            } else {
              $merged[$type] = array_merge($merged[$type], $list);
            }
          }
        } else {
          throw new EUverseWiki("Invalid $settings->attachments value: $merge.");
        }
      }

  function ToHTML() { return $this->RenderIn('html'); }
  function ToHTML5() { return $this->RenderIn('html5'); }

  function SerializeTo($handle, array $options = array()) {
    $options += array('serializer' => null, 'flags' => 0, 'warningCallback' => null,
                      'gzLevel' => null, 'filterCallback' => null);

      isset( $options['gzLevel'] ) and $options['flags'] |= UWikiSerializer::GZ;
      isset( $options['filterCallback'] ) and $options['flags'] |= UWikiSerializer::Filter;

    $ser = new UWikiSerializer($handle, $options['flags']);

      $ser->warningCallback = $options['warningCallback'];
      $ser->filterCallback = $options['filterCallback'];
      isset( $options['serializer'] ) and $options['serializer'] = $ser;

    $ser->WriteHeader();
    $ser->WriteDocument($this);
    $ser->Finish();
  }

    function WriteTo(UWikiSerializer $ser) {
      $this->WriteHeaderTo($ser);
      $ser->WriteSettings($this->settings);
      $this->WriteRootTo($ser);
    }

      // Header structure:  [W]  version * 1000
      //                    [W]  build number
      //                    [B]  type of $this->title - 'n'ull, 'e'lement, 's'tring
      //                    [W]  document ID
      //                    then Serializer's header
      //                    [A]  metadata ($this->meta)
      //                    [A]  attachments ($this->attachments)
      //                     ?   title element or string unless it's null
      function WriteHeaderTo(UWikiSerializer $ser) {
        if ($this->title === null) {
          $titleType = 'n';
        } elseif ($this->title instanceof UWikiBaseELement) {
          $titleType = 'e';
        } elseif (is_string($this->title)) {
          $titleType = 's';
        } else {
          $ser->Error('wrong value of document\'s object $title property - must be'.
                      ' a string, null or an UWikiBaseELement, got a '.gettype($this->title));
        }

        $ser->Pack('vvav', $this->version * 1000, $this->build, $titleType, $this->ID());
        $ser->WriteArray($this->meta);
        $ser->WriteArray($this->attachments);

        switch ($titleType) {
        case 'e': $ser->WriteElement($this->title); break;
        case 's': $ser->WriteString($this->title); break;
        }
      }

      function WriteRootTo(UWikiSerializer $ser) { $ser->WriteElement($this->root); }

  // $handle - resource or an instance of UWikiUnserializer.
  static function UnserializeFrom($handle, array $options = array()) {
    $options += array('unserializer' => null, 'warningCallback' => null,
                      'filterCallback' => null, 'settings' => null);

    $ser = new UWikiUnserializer($handle);

      $ser->warningCallback = $options['warningCallback'];
      $ser->filterCallback = $options['filterCallback'];
      $options['unserializer'] and $options['unserializer'] = $ser;

    $ser->ReadHeader();
    $doc = $ser->ReadDocument($options);
    $ser->Finish();

    return $doc;
  }

    static function ReadFrom(UWikiSerializer $ser, &$doc, $class, array $options) {
      $options += array('settings' => null);

      self::ReadHeaderFrom($ser, $doc, $class);

        if ($settings = $options['settings']) {
          $doc->settings = $settings;
          $ser->ReadSettingsInto($doc->settings);
        } else {
          $doc->settings = $ser->ReadSettings();
        }

        $doc->ReadRootFrom($ser);

      $ser->LeaveDoc($doc);
    }

      static function ReadHeaderFrom(UWikiUnserializer $ser, &$doc, $class) {
        if (!class_exists($class)) {
          $ser->Error("document class $class is undefined; input stream might not".
                      ' contain an UverseWiki document');
        }

        $header = $ser->Unpack('vversion/vbuild/atitleType/vid', 7);

        $doc = new $class('');

          if (! $doc instanceof self) { $ser->Error("document class $class is not a child of ".__CLASS__); }
          $doc->id = $header['id'];
          $ser->CurrentDoc($doc);

        $doc->version = $header['version'] / 1000;
        $doc->build = $header['build'];

        if ($doc->build > UverseWikiBuild) {
          $msg = sprintf('serialized document is of more recent version (%1.1f, build'.
                         ' R%d) then loaded UverseWiki framework (version %1.1f, build R%d)',
                         $doc->version, $doc->build, UverseWikiVersion, UverseWikiBuild);
          $ser->Warning('newer doc version', $msg);
        }

        $doc->meta = $ser->ReadArray();
        $doc->attachments = $ser->ReadArray();

        switch ($header['titleType']) {
        case 'n':  $doc->title = null; break;
        case 'e':  $doc->title = $ser->ReadElement(); break;
        case 's':  $doc->title = $ser->ReadString(); break;
        default:   $ser->Warning('unknown title type', "unknown \$titleType value - must be 'n', 'e' or 's' but got '$titleType'");
        }

        return $doc;
      }

      function ReadRootFrom(UWikiUnserializer $ser) {
          $root = $ser->ReadElement();
          $this->LoadMarkup( self::MarkupNameFrom($root->elementName) );
          $this->root = $root;
      }

  function &Serialize(array $options = array()) {
    $h = fopen('php://temp', 'w+b');

      $this->SerializeTo($h, $options);
      $result = stream_get_contents($h, -1, 0);

    fclose($h);
    return $result;
  }

    static function Unserialize($str, array $options = array()) {
      $h = fopen('php://temp', 'w+b');

        fwrite($h, $str, strlen($str));
        $str = null;
        rewind($h);

        $doc = self::UnserializeFrom($h, $options);

      fclose($h);
      return $doc;
    }


  /* Utility methods: */

  // returns true if one of callbacks has returned loose true.
  static function CallAll($callbacks, array $args = array()) {
    if (is_array($callbacks)) {
      foreach ($callbacks as $callback) {
        if (self::Call($callback, $args)) { return true; }
      }
    }
  }

    static function Call($callback, array $args = array()) {
      $args = (array) $args;

      if (is_array($callback) and count($callback) > 2) {
        $callArgsFirst = $callback[1][0] === '*';
        if ($callArgsFirst) {
          $callArgs = array_merge(array_splice($callback, 2), $args);
        } else {
          $callArgs = array_merge($args,  array_splice($callback, 2));
        }
      } else {
        $callArgs = $args;
      }

      is_array($callback) and $callback[1] = ltrim($callback[1], '*');
      return call_user_func_array($callback, $callArgs);
    }

    static function CanCall($callback) {
      if ($callback) {
        if (is_array($callback) and isset($callback[1])) {
          $callback = array_slice($callback, 0, 2);
          $callback[1] = ltrim($callback[1], '*');
        }
        return is_callable($callback);
      }
    }

  static function Deprecated($feature, $issuer = null, $desc = null) {
    $exception = new EUWikiDeprecated($feature, $issuer, $desc);

    switch ($rep = self::$deprecatedReporting) {
    case 'exception':   throw $exception; break;
    case 'message':     echo $exception; break;
    case false:         break;
    default:
      if (is_callable($rep)) {
        $rep($exception);
      }
    }
  }

  function NewElement($class) {
      $aliasedTo = &$this->settings->classAliases[$class];
      $aliasedTo and $class = $aliasedTo;

    $obj = new $class;
    $obj->doc = $this;
    $obj->settings = &$this->settings;
    $obj->strings = &$this->settings->strings;
    $obj->pager = $this->settings->pager;
    $obj->style = $this->style;
    $obj->elementID = $this->settings->GiveIdTo($obj);

    $this->elements[$obj->elementName][] = $obj;

    if ($handlers = &$this->newElementHooks[$obj->elementName]) {
      self::CallAll($handlers, array(&$obj));
    }
    if ($handlers = &$this->newElementHooks['any']) {
      self::CallAll($handlers, array(&$obj));
    }

    return $obj;
  }

  function NewLinkedDocument($source = '', $markupName = null) {
    $doc = new self($source);
    $doc->settings = $this->settings->LinkedClone();

    // hooks must be called before LoadMarkup() so that it's possible to set hooks
    // like on new element creation (LoadMarkup() creates root element).
    self::CallAll($this->newDocumentHooks, array(&$doc, $this));

    $markupName === null or $doc->LoadMarkup($markupName);
    return $doc;
  }

  // can return null if no title was assigned, heading element if it sets the title or
  // string if title is passed externally and it's a string (UWikiDocument->$title).
  function GetTitle() {
    foreach ($this->settings->headings as $obj) {
      if ($obj->level === 1) { return $obj; }
    }

    return $this->title;
  }

    function HtmlTitle() {
      $title = $this->GetTitle();

      if (is_object($title)) {
        $this->BeginRenderingInto('html');
        $title = $title->ChildrenToHTML(false);
        $this->EndRenderingInto('html', $title);
      }

      return "$title";
    }

    function TextTitle() {
      $title = $this->GetTitle();
      return ($title and is_object($title)) ? strip_tags( $this->HtmlTitle() ) : "$title";
    }

  function InlineHTML($ifBlocks = 'message') {
    $this->BeginRenderingInto('html');
    $result = $this->root->InlineHTML($ifBlocks);
    $this->EndRenderingInto('html', $result);

    return $result;
  }


  /* API methods */

  static function Quote($str, $markup) {
    self::CallAll(self::$quoteHooks, array(&$str, &$markup));
    return $str;
  }

    // default handler for Quote('...', 'wacko').
    protected static function QuoteWacko(&$str, &$markup) {
      if ($markup === 'wacko') {
        defined('Uwacko_TildeToken') or self::PreloadMarkup('wacko');

        // str""str -> str~""str           str~""str -> str~~~""str
        // str~~""str -> str~~~""str       str~~~""str -> str~~~~ ~""str
          $t = Uwacko_TildeToken;
          $q = Uwacko_SubstrEscToken;
        $str = preg_replace("/(?<!$t)($t?)($t$t)*($q)/u", '\1\2\1\2'.$t.'\3', $str);
        EUWikiLastPCRE::ThrowIfPcreFailed();

        $str = '""'.$str.'""';
      }
    }

  // $doc - UWikiDocument or a string. returns either a UWikiBaseElement, a string or null.
  static function GetTitleOf($doc, $markup, $quick = true) {
    self::CallAll(self::$getTitleHooks, array(&$title, &$doc, &$markup, &$quick));
    return "$title" === '' ? null : $title;
  }

    static function GetTitleOfCore(&$gotTitle, &$doc, &$markup, &$quick) {
      static $regexps = array('wacko' => '/(\n|^)==([^=].*?)\s*==(\r?\n|$)/u',
                              'html'  => '/<title>(.+)<\/title>/su');

      if ($quick and !($doc instanceof self)) {
        $regexp = &$regexps[$markup];
        if ($regexp) {
          if (preg_match($regexp, $doc, $matches)) {
            $gotTitle = trim($matches[2]);
            return true;
          } else {
            EUWikiLastPCRE::ThrowIfPcreFailed();
          }
        }
      } else {
        $doc = ($doc instanceof self) ? $doc : self::TryParsing($doc, null, $markup);
        if ($doc) {
          $gotTitle = $doc->GetTitle();
          return true;
        }
      }
    }

  function ID() {
    $this->id or $this->id = $this->settings->GiveIdTo($this);
    return $this->id;
  }

  function Attach($type, $data, $name = null) {
    $name === null ? ($this->attachments[$type][] = $data) : ($this->attachments[$type][$name] = $data);
  }

    function MergeAttachmentsOfNestedDocs() { $this->settings->attachments = $this; }
}

UWikiDocument::Initialize();
