<?php
include_once 'inline.php';

  self::$loadedHandlers[$markup]['block'][] = array('(?<=\r\n(?:\r\n)+)', 'Uwacko_TextBlock', 0);
  self::$loadedHandlers[$markup]['block'][] = array('(?<=\n(?:\n)+)', 'Uwacko_TextBlock', 0);
  self::$loadedHandlers[$markup]['block'][] = array('\n\s* (?= \.\([^\)\n]+\) )', 'Uwacko_TextBlock', 0);
  self::$loadedHandlers[$markup]['block'][] = array('(?<=\r\n|\n) \s* (?: >>|<<)', 'Uwacko_TextBlock', Uwacko_StartTag | Uwacko_Callback | Uwacko_DiscardOnDiffToken);

  self::$loadedHandlers[$markup]['block'][] = array('(?: >>|<<) (?:\r?\n)+', 'Uwacko_TextBlock', Uwacko_EndTag | Uwacko_Callback);

  // fmt: 'Class' => handler priority (0 - lowest, 100 - highest). If you add to this make sure to arsort() it.
  self::$loadedHandlers[$markup]['line']['Uwacko_ParagraphLine'] = 0;

class Uwacko_TextBlock extends Uwacko_Base {
  static function FindTokenCallback($doc, &$raw, &$positions, &$stack, &$token, &$pos, &$flags) {
    // New alignment token >>/<< discards previous and starts anew.
    if (($flags & Uwacko_EndTag) == 0 and !empty($stack) and $stack[0][0] === __CLASS__) {
      $positions[$pos] = array_pop($positions);
      $stack[0][1] = $stack[0][4] = $pos;
      return Uwacko_SkipThisToken;
    } else {
      if ($flags & Uwacko_EndTag) { $pos += strlen($token); }
      $token = '';  // token must not be excluded from text; is handled by Uwacko_Paragraph.
    }
  }

  function Parse() {
    parent::Parse();

    $lines = explode("\n", $this->raw);

    $lineObjects = array();
      while (($line = array_shift($lines)) !== null) {
          $line = rtrim($line, "\r");
          if ($line === '') { continue; }

        $class = $this->LineClassFor($line);
        $lineObjects[] = $obj = $this->NewElement($class);
        $obj->SetRawAndSource($line);

        while ($lines and ($breakChar = $obj->ContinuesIf($lines[0], $this->doc)) !== false) {
          is_string($breakChar) and $obj->raw .= $breakChar;
          $obj->raw .= rtrim( array_shift($lines) );
        }
      }

    $this->FindLineQuotesIn($lineObjects);

    $this->children = array();
    return $this->PutLines($lineObjects);
  }

    function LineClassFor(&$line) {
      $handlers = $this->settings->handlers->Get('line');

      foreach ($handlers as $class => $priority) {
        $result = call_user_func(array($class, 'CanHandleLine'), $line, $this->doc);

        if ($result === true) {
          return $class;
        } elseif (is_string($result)) {
          // it was escaped, example:   ~> Line quote.
          $line = $result;
          return self::LastIn($handlers);
        }
      }
    }

    // Resolves ambiguity of Line quote >> and paragraph alignment >>..*.
    function FindLineQuotesIn(&$lines) {
      $linesToReplace = array();

      $quoteIndex = null;

      foreach ($lines as $i => $obj) {
        if ($obj->className === 'Uwacko_ParagraphLine') {
          $canHandle = Uwacko_QuoteLine::CanHandleLine($obj->raw, $this->doc);

          if ($canHandle ) {
            // >>\n>>... - no ending tag for the first >> thus it was a line quote.
            $linesToReplace[] = $quoteIndex;
            $quoteIndex = $i;

            if (is_string($canHandle)) {
              // It's escaped (see LineCLassFor) which means alignment, if any, is also escaped.
              // If that's the case we leave one tilde behind so the para will escape it
              // but first check and see if it's really aligned.
              $escapedRaw = $obj->raw;
              $obj->raw = $canHandle;
            }
          }

            $aligned = false;

          if ($quoteIndex !== null) {
            $possiblePara = array_slice($lines, 0, $i + 1);
            if (Uwacko_Paragraph::GetAlignOf($possiblePara, true)) {
              // >>\n...<< - ending tag found, it was an aligned paragraph.
              $obj->forceNewGroup = !is_string($canHandle);
              $quoteIndex = null;
              $aligned = true;
            }
          }

            if (is_string($canHandle) and !$obj->forceNewGroup) {
              // ~>>para<<  - both 2nd level inline quote & ">><<" alignment are escaped.
              $aligned and $obj->raw = $escapedRaw;
              $quoteIndex = null;
            }
        } else {
          $linesToReplace[] = $quoteIndex;
          $quoteIndex = null;
        }
      }
        // >>... [EOF]
        $linesToReplace[] = $quoteIndex;

      foreach ($linesToReplace as $i) {
        if ($i !== null) {
          $quoteObj = $this->NewElement('Uwacko_QuoteLine');
          $quoteObj->SetRawAndSource( $lines[$i]->raw );
          $lines[$i] = $quoteObj;
        }
      }
    }

  function PutLines($lines) {
    $children = array();

      if ($lines) {
        $class = $lines[0]->groupClass;
        $children[] = $groupParent = $this->NewElement($class);
        $groupParent->treePosition = $this->treePosition;

        foreach ($lines as $i => $obj) {
            $nextClass = isset( $lines[$i + 1] ) ? $lines[$i + 1]->groupClass : null;

          $groupParent->children[] = $obj;
          $obj->isFirst = ($i === 0 or $lines[$i - 1]->isLast);
          $obj->isLast = $nextClass !== $class;

          if ($nextClass !== $class or $obj->forceNewGroup) {
            $groupParent->ParseLines();

            if ($nextClass != null) {
              $class = $lines[$i + 1]->groupClass;
              $children[] = $groupParent = $groupParent->NewElement($class, 'sibling');
            }
          }
        }
      }

    return $children;
  }
}

abstract class Uwacko_Lines extends Uwacko_Base {
  public $isBlock = true;

  function ParseLines() {
    $this->SetSettingsFromLines($this->children);
    foreach ($this->children as $obj) { $obj->Parse(); }
  }

    function SetSettingsFromLines(&$lines) { }
}

class Uwacko_Paragraph extends Uwacko_Lines {
  static $alignChars = array('<<<<' => 'left', '>>>>' => 'right', '>><<' => 'center', '<<>>' => 'justify');

  public $kind = 'paragraph';
  public $htmlTag = 'p';

  public $indentLevel;
  public $userStyles = array();
  public $align;
    public $explicitAlign = false;

  function SetupSerialize(array &$props) {
    parent::SetupSerialize($props);

    $props['int'][] = 'indentLevel';
    $props['list'][] = 'userStyles';
    $props['str'][] = 'align';
    $props['bool'][] = 'explicitAlign';
  }

  function SetSettingsFromLines(&$lines) {
    $this->SetIndentLevelFrom($lines);
    $this->SetAlignFrom($lines);
    $this->SetStyleFrom($lines);
  }

    function SetIndentLevelFrom(&$lines) {
      $this->indentLevel = 0;

        if (!isset( $lines[1] ) and
            $this->settings->noLineBreaksInsideParagraph and
            $this->settings->moreIndentBreaksParaLine) {
          // special: no single-line paragraphs indentation in Novel mode; see docs.
          return;
        }

      $minLeadingSpaces = 0xFFFF;

        foreach ($lines as $obj) {
          $thisIndent = $this->CountLeftWhiteSpace($obj->raw);
          if ($thisIndent >= 2) {
            $minLeadingSpaces = min($minLeadingSpaces, $thisIndent);
          } else {
            return;
          }
        }

      $this->indentLevel = (int) ($minLeadingSpaces / 2);

        foreach ($lines as $obj) {
          $obj->SetRawAndSource( substr($obj->raw, $minLeadingSpaces) );
        }
    }

    function SetAlignFrom(&$lines) {
      $align = self::GetAlignOf($lines, true);

        $firstLine = &$lines[0]->raw;
        $this->explicitAlign = $align != null;

        if ($align) {
          $firstLine = substr(ltrim($lines[0]->raw), 2);
          self::DeleteSubstrIn( self::LastIn($lines)->raw, -2, 2 );
        } else {
          $whitespaceEnd = self::CountLeftWhiteSpace($firstLine);
          $tildeCount = self::CountLeftCharIn($firstLine, Uwacko_TildeToken, $whitespaceEnd);
          if ($tildeCount % 2 !== 0) {
            $leftAlign = substr($firstLine, $whitespaceEnd + $tildeCount, 2);
            if (isset(self::$alignChars[ $leftAlign.$leftAlign ])) {
              self::DeleteSubstrIn($firstLine, $whitespaceEnd);
            }
          }
        }

      $this->align = $align ? $align : 'left';
    }

      static function GetAlignOf(&$lines, $deleteEscaping = false) {
          $first = ltrim($lines[0]->raw);
          $last = &self::LastIn($lines)->raw;

        if (count($lines) == 1 ? strlen($first) > 4 : ( strlen($first) > 2 and strlen($last) > 2 )) {
          if ($alignment = &self::$alignChars[ substr($first, 0, 2) . substr($last, -2) ]) {
            $tildeCount = 0;
            $i = strlen($last) - 2;
              while ($last[--$i] === Uwacko_TildeToken) { ++$tildeCount; }

            if ($tildeCount % 2 === 0) {
              return $alignment;
            } elseif ($deleteEscaping) {
              self::DeleteSubstrIn($last, -3);
            }
          }
        }
      }

    // .(style)   - leading dot can be omitted if any align was specified.
    function SetStyleFrom(&$lines) {
      $styles = array();
      $raw = &$lines[0]->raw;

        if (isset( $raw[2] ) and $raw[0] === Uwacko_TildeToken
            and $raw[1] !== Uwacko_TildeToken ) {
          $rawWithoutStyles = $raw;
          $styles = $this->ExtractStylesFrom($rawWithoutStyles, 1);

          if ($styles) {
            $raw = substr($raw, 1);
            $styles = array();
          }
        } else {
          $styles = $this->ExtractStylesFrom($raw);
        }

      if ($raw === '') {
        isset($lines[1]) and array_shift($lines);
        $lines[0]->isFirst = true;
      }

      $styles or $styles[] = null;  // to get the default style.
      foreach ($styles as $style) {
        $this->htmlClasses[] = $this->userStyles[] = $this->RealStyleBy($style);
      }
    }

      function ExtractStylesFrom(&$raw, $offset = 0) {
        if ($raw[$offset] === '.' or $this->explicitAlign) {
          $prefixLength = (int) ($raw[$offset] == '.');
          return $this->style->ExtractMultipleFrom($raw, $offset + $prefixLength);
        } else {
          return array();
        }
      }

  function SelfToHtmlWith($contents) {
    $html = parent::SelfToHtmlWith($contents);
      $footnotes = $this->settings->followingFootnotes;
      $this->settings->followingFootnotes = array();
      foreach ($footnotes as $obj) { $html .= $obj->AllToHTML(); }
    return $html;
  }

    function SelfHtmlAttributes() {
      $attrs = array();

        $this->align == 'left' or $attrs['style'] = 'text-align: '.$this->align;
        $this->indentLevel and $attrs['class'] = 'indent-'.$this->indentLevel;

      return $attrs;
    }
}

abstract class Uwacko_Line extends Uwacko_InlineElement {
  public $isBlock = true;

  public $isFirst;        // is set by Uwacko_TextBlock.
  public $isLast;         // is set by Uwacko_TextBlock.
  public $forceNewGroup;  // is set and used only by Uwacko_TextBlock->FindLineQuotesIn().

  public $groupClass;

  function SetupSerialize(array &$props) {
    parent::SetupSerialize($props);
    array_push($props['bool'], 'isFirst', 'isLast', 'forceNewGroup');
  }

  function HasBlockChildren() { return false; }

  static function CanHandleLine($line, $doc) {
    throw new EUverseWiki('Uwacko_Line::CanHandleLine() must be overriden.');
  }

  function ContinuesIf($newLine, $doc) { return false; }
}

class Uwacko_ParagraphLine extends Uwacko_Line {
  public $isSingleLineHTML = true;

  public $kind = 'line';
  public $groupClass = 'Uwacko_Paragraph';

  function SelfToHtmlWith($contents) {
    if (!$this->settings->noLineBreaksInsideParagraph) {
      $this->isLast or $contents .= "<br />\n<span class=\"soft-break\"></span>";
    } elseif ($this->settings->moreIndentBreaksParaLine and !$this->isFirst
              and $this->CountLeftWhiteSpace($this->raw) >= 2) {
      $contents = "<br />\n<span class=\"more-ind-break\"></span>$contents";
    } else {
      $this->isLast or $contents .= "\n";
    }

    return parent::SelfToHtmlWith($contents);
  }

  static function CanHandleLine($line, $doc) { return true; }
}

class Uwacko_QuoteLine extends Uwacko_Line {
  public $groupClass = 'Uwacko_QuoteLines';
  public $level;

  function SetupSerialize(array &$props) {
    parent::SetupSerialize($props);
    $props['int'][] = 'level';
  }

  function SetSettingsFrom(&$raw) {
      $raw = ltrim($raw);

    $level = self::CountLeftCharIn($raw, '>');
    $raw = substr($raw, $level);
    $this->level = min($level, $this->settings->maxLineQuoteLevel);
  }

  function SelfToHtmlWith($contents) {
      $classes = $this->settings->globalHtmlClasses;
      $classes[] = 'level-'.$this->level;
    $html = '<li class="'.join(' ', $classes).'"><cite>'.$contents.'</cite></li>';
    return parent::SelfToHtmlWith($html);
  }

  static function CanHandleLine($line, $doc) {
      $line = ltrim($line);

    if (isset($line[2]) and $line[0] === Uwacko_TildeToken
       and $line[1] !== Uwacko_TildeToken) {
      $level = self::CountLeftCharIn(substr($line, 1), '>');
      $maybeEscaped = true;
    } else {
      $level = self::CountLeftCharIn($line, '>');
    }

      $level <= $doc->settings->maxLineQuoteLevel or $level = 0;

    if ($level) {
      // the caller will recognize string as "I could handle but it was escaped".
      return isset($maybeEscaped) ? substr($line, 1) : true;
    }
  }
}

  class Uwacko_QuoteLines extends Uwacko_Lines {
    public $htmlTag = 'blockquote';
    public $htmlClasses = array('inline');

    function SelfToHtmlWith($contents) {
      return parent::SelfToHtmlWith("<ol>\n$contents</ol>");
    }
  }
