<?php
/*
  This action lets you collect statistics about links used in a document (or documents).
  It can either autoload (disabled by default) or explicitly called for current document.

  You might also be interested in {{LinkProxy}} action - can provide other features like
  adding rel="nofollow" to external links.

  Each call to {{LinkStat}} adds one rule; rules are applied in order they were defined
  to each ((link)) in the document. Multiple rules can apply to a link if they match.
  {{LinkStat}} without arguments will enable this action for current doc with default
  set of rules (LinkStatRules). {{LinkStat clear}} will clear current rules.

  {{LinkStat pattern, exclude=pattern, match=indexlist, read=indexlist, tooltip=str, proxy=str,
             inc=indexlist, storage=str, breaks}}
    = pattern   A regular expression snippet; brackets will be auto-added unless it starts
      with "/" (in this case you can supply custom modifiers, e.g.: "/regexp/i").
    = exclude   A regular expression that is matched after matching 'pattern'; same rules
      for regexp brackets apply as per 'pattern'.
    = match     Index-list (see below) of match indexes to pass to tooltip; defaults to "false".
      Constants (non-numbers) are passed to the tooltip as is. Indexes outside of matches are '0'.
    = read      Set of index-list (see below) to pass to tooltip as format args; defaults to "false".
    = tooltip   If a language string with this name (in strings.conf) exists, it's used;
      otherwise a string named "{{linkstat: " + this value is searched; ultimately the value
      itself is used if nothing else found. Gets passed 'match'es and 'read's (see above).
      If this starts with "+" tooltip is appended, otherwise - set to. Format strings are:
      1. @ - fetches next value off the 'match' parameter (exception if none left).
      2. $ - same as @ but for 'read' parameter.
      Also, the value of 'tooltip' can be a callback function if it starts with 'LinkStatCallback'
      (and it can have nothing else following). Its format: function ($matches, $reads).
    = proxy     Changes link to this pattern - useful which tracking number of downloads.
      The value of "0" disables URL modification; other turns it on. Format strings are:
      1. {URL}                - URL-encoded original link; it's automaticaly appended if
                                proxy URL contained none of the format strings.
      2. {RAW}                - raw original link.
      3. {Index} (e.g. {-1})  - is replaced by the value of given match pocket.
    = inc       Set of index-list (see below); specifies if and where to store number of
      occurrences of this rule (each time summed up with previous value - on each document
      render). If passed as flag (just "inc") defaults to "-1"; if omitted - to "false".
    = storage   This switches storage interface by its name in LinkStatStorages. If it's
      invalid "tooltip" and "inc" parameters will be ignored. Defaults to 'default'.
    = breaks If this rule has matched all following rules will be skipped for the current link.

  An "index-list" is either "false" (or "no") or a string. If it's a string, it's a space-separated
  sequence of numbers and strings. Numbers refer to match pocket ("0" refers to the full match
  and a special value of -1 (which is the default) refers to the entire URL), string is a
  constant (if it starts with "_" AND has at least one "_" later the first "_" is removed
  and all others are replaced with spaces; if it starts with "__" the first "_" is removed
  but all others are left intact).
  "Sets of index-list" are just arrays of index-lists separated by "|". Example: "read=-1 | downloads 2".
  For 'read' individual "false" index-lists result in string '0', for 'inc' they're ignored.

  By using different combinations of parameters in one rule you can achieve interesting effects.

  For example, to add "File X was downloaded N times" to each RAR or ZIP link use this rule:
    {{LinkStat ([^\\/]+\.(rar|zip))$, match=1, read=downloads -1, tooltip=download, proxy="dl.php?{URL}"}}
  This rule requires dl.php script to increment actual download counter (which is simple
  thanks to base Ulinkstat_BaseStorage class). Other parameters here:
    * match & read - provides format arguments for tooltip;
    * tooltip - adds "title=" HTML attribute from language string "download" or
                "{{linkstat: download" - whichever exists (for example, "@ was downloaded $ times.").
    * proxy - changes each matching URL to the form like "dl.php?my+app.rar".

  You can count the number of all external links in your document(s) when they're rendered
  (for whatever the reason), anonymize them and add a tooltip:
    {{LinkStat ^http://, proxy=http://anonymize/, read=external -1, inc=external -1
               tooltip=This is external link number $.}}

  CONFIGURATION - fields added to UWikiSettings:

    * LinkStatStorages - an array of storage interfaces (object instances). Should contain
      at least one member named 'default', otherwise 'tooltip' and 'store' attributes are
      ignored. Initially it's empty so make sure to set it yourself, such as like this:
        $storage = new Ulinkstat_FileStorage(UWikiRootPath.'/linkstat.php');
        $settings->LinkStatStorages = array('default' => $storage);
    * LinkStatRules - set of default rules. To clear it write {{LinkStat clear}}. The format:
      array( array('pattern' => '\.rar', 'tooltip' => 'RAR archive'), array(...), ... )
      WARNING: DO NOT ADD RULES TO THIS PROPERTY DIRECTLY - instead call Ulinkstat_Root::AddRule().
    * LinkStatCurrentRules Array of normalized rules. WARNING: DO NOT MODIFY IT IN ANY WAY.
      It, however, might prove useful because each rule maintains a counter applyCount.
*/

UWikiDocument::$afterParsingHooks[] = array('Ulinkstat_Root', 'Autoload');

class Ulinkstat_Root extends UWikiBaseAction {
  public $parameterOrder = array('pattern');

  function IsDynamic() { return true; }

  function Execute($format, $params) {
    self::EnableFor($format->settings);

    if (!empty($params)) {
      if ($params['pattern'] === 'clear') {
        $format->settings->LinkStatCurrentRules = array();
      } else {
        self::AddRuleTo($format->settings->LinkStatCurrentRules, $params);
      }
    }
  }

    static function ChangeURL(&$url, $linkObj) {
      foreach ($linkObj->settings->LinkStatCurrentRules as $rule) {
        if (preg_match($rule['pattern'], $url, $match) and
            (!$rule['exclude'] or !preg_match($rule['exclude'], $url, $match))) {
          self::ApplyRuleTo($url, $linkObj, $rule, $match);
          if ($rule['breaks']) { return; }
        }
      }
    }

      static function ApplyRuleTo(&$url, $linkObj, &$rule, &$match) {
        ++$rule['applyCount'];

        if ($storage = self::GetStorage($rule['storage'], $linkObj->settings)) {
          if ($name = $rule['tooltip']) {
            $matches = array();
            foreach ($rule['match'] as $index) {
              $matches[] = self::MatchAt($index, $url, $match);
            }

            $reads = $storage->ReadMany( self::IndexSetToPaths($rule['read'], $url, $match), 0 );

            $tip = self::GetTooltip(ltrim($name, '+'), $matches, $reads, $linkObj->settings->strings);

            $s = &$linkObj->passedURL;
            $name[0] === '+' or $s = '';
            $s === '' or $s .= ' ';
            $s .= $tip;
          }

          $storage->IncrementMany( self::IndexSetToPaths($rule['inc'], $url, $match) );
        }

        if ($rule['proxy'] !== '') {
          $url = self::ProxyURL($rule['proxy'], $url, $match);
        }
      }

        static function MatchAt($index, $url, &$matches) {
          return $index === -1 ? $url : ( isset($matches[$index]) ? $matches[$index] : '' );
        }

        static function GetTooltip($name, $matches, $reads, &$strings) {
          if (strpos($name, 'LinkStatCallback') === 0 and function_exists($name)) {
            return $name($matches, $reads);
          } else {
            if (isset($strings[$name])) {
              $name = $strings[$name];
            } elseif (isset($strings["{{linkstat: $name"])) {
              $name = $strings["{{linkstat: $name"];
            }

            $name = self::SimpleFmt($name, $matches, '@');
            $name = self::SimpleFmt($name, $reads, '$');
            return $name;
          }
        }

        static function ProxyURL($proxyURL, $linkURL, &$match) {
          $url = strtr($proxyURL, array('{URL}' => urlencode($linkURL), '{RAW}' => $linkURL));

          if (strpos($url, '{') !== false and preg_match_all('/\{(\d+)\}/', $url, $matches)) {
            $replace = array();

              foreach ($matches[1] as $index) {
                $replace["{$index}"] = self::MatchAt($index, $linkURL, $match);
              }

            $url = strtr($url, $replace);
          }

          return $url === $proxyURL ? $url.urlencode($linkURL) : $url;
        }

        static function GetStorage($name, $settings) {
          $name or $name = 'default';
          $obj = $settings->LinkStatStorages[$name];
          return $obj instanceof Ulinkstat_BaseStorage ? $obj : null;
        }

        static function IndexSetToPaths($indexSet, $url, &$match) {
          $paths = array();

            foreach ($indexSet as $list) {
              $path = array();
              foreach ($list as $index) {
                if (is_int($index)) {
                  $path[] = self::MatchAt($index, $url, $match);
                } else {
                  if ($index[0] === '_') {
                    $index = substr($index, 1);
                    if ($index[1] !== '_' and strpos($index, '_', 1)) {
                      $index = strtr($index, '_', ' ');
                    }
                  }

                  $path[] = $index;
                }
              }
              $paths[] = $path;
            }

          return $paths;
        }

  static function AddRuleTo(&$array, $rule) {
    $rule += array('pattern' => '', 'exclude' => '', 'match' => '', 'read' => '', 'tooltip' => '',
                   'proxy' => '', 'inc' => '', 'storage' => '', 'breaks' => false);

    self::NormalizeRegExp($rule['pattern']);
    self::NormalizeRegExp($rule['exclude']);

    $rule['match'] = self::ParseIndexList($rule['match'], true);
    self::ParseIndexSet($rule['read']);
    self::ParseIndexSet($rule['inc']);

    $rule['applyCount'] = 0;
    $array[] = $rule;
  }

    static function NormalizeRegExp(&$regexp) {
      if ($regexp !== '' and $regexp[0] !== '/') {
        $regexp = '~'.str_replace('~', '\\~', $regexp).'~';
      }
    }

    static function ParseIndexSet(&$str) {
      $pieces = explode('|', $str);
      $str = array();

      foreach ($pieces as $piece) {
        if (($piece = trim($piece)) !== '') {
          $str[] = self::ParseIndexList($piece);
        }
      }
    }

    static function ParseIndexList(&$str) {
      if (in_array(strtolower($str), array('false', 'no'))) {
        $str = null;
      } else {
        $pieces = explode(' ', $str);
        $str = array();

        foreach ($pieces as $piece) {
          if (($piece = trim($piece)) !== '') {
            $str[] = is_numeric($piece) ? ((int) $piece) : $piece;
          }
        }
      }

      return $str;
    }

  static function Autoload($doc) {
    empty($doc->settings->LinkStatAutoload) or self::EnableFor($doc->settings);
  }

    static function EnableFor($settings) {
      if (empty($settings->LinkStat)) {
        $settings->LinkStat = true;
        $settings->LinkStatCurrentRules = isset($settings->LinkStatRules) ? $settings->LinkStatRules : array();
        $settings->handlers->AddOrReplace('linkstat', 'linkOnRender', array(__CLASS__, 'ChangeURL'));
      }
    }
}

  class Ulinkstat_NoURL extends EUverseWiki {
    function __construct() {
      parent::__construct('{{LinkStat}} should be either passed URL or has LinkStatURL set in settings.');
    }
  }

  class Ulinkstat_StorageError extends EUverseWiki {
    function __construct($obj, $info) { parent::__construct(get_class($obj).': '.$info); }
  }

abstract class Ulinkstat_BaseStorage {
  // $keys: array( array('key', 'subkey', 'subsubkey', ...), array('key2', ...), ... )
  function ReadMany($paths, $default = null) {
    $results = array();
    foreach ($paths as $path) {
      $result = $this->Read($path);
      $results[] = $result === null ? $default : $result;
    }
    return $results;
  }

    function Read($path) {
      $current = $this->ReadKey(array_shift($path));

      while ($piece = array_shift($path)) {
        if (isset($current) and !is_array($current)) {
          throw new Ulinkstat_StorageError($this, 'cannot read by path - current isn\'t an array.');
        }
        $current = &$current[$piece];
      }

      return $current;
    }

  function Write($path, $value) {
    $key = array_shift($path);
    $root = $this->ReadKey($key);

      $current = &$root;
      while ($piece = array_shift($path)) {
        if (isset($current) and !is_array($current)) {
          throw new Ulinkstat_StorageError($this, 'cannot write by path - current isn\'t an array.');
        }
        $current = &$current[$piece];
      }
      $current = $value;

    $this->WriteKey($key, $root);
  }

  function IncrementMany($paths) {
    foreach ($paths as $path) { $results[] = $this->Increment($path); }
  }

    function Increment($path) {
      $current = $this->Read($path);
      if (isset($current) and !is_numeric($current)) {
        throw new Ulinkstat_StorageError($this, 'cannot increment - not a number.');
      }

      $this->Write($path, ++$current);
      return $current;
    }

  abstract function ReadKey($key);
  abstract function WriteKey($key, $value);
}

  class Ulinkstat_FileStorage extends Ulinkstat_BaseStorage {
    public $file;

    function __construct($file) { $this->file = $file; }

    function ReadKey($key) {
      $tree = $this->ReadRoot();
      return isset($tree[$key]) ? $tree[$key] : null;
    }

    function WriteKey($key, $value) {
      $tree = $this->ReadRoot();
      $tree[$key] = $value;
      $this->WriteRoot($tree);
    }

    function ReadRoot() { return is_file($this->file) ? include($this->file) : array(); }

    function WriteRoot($tree) {
      $export = "<?php\nreturn ".var_export($tree, true).';';
      if (!is_int( file_put_contents($this->file, $export, LOCK_EX) )) {
        throw new Ulinkstat_StorageError($this, 'cannot write '.$this->file);
      }
    }
  }
