<?php
/*
  A web 2.0 %%mirror. Creates a textarea with given markup (wacko or other) and
  when user hovers over it shows a small window with formatted view next to the
  source (this requires neither JavaScript nor AJAX).

  It allows editing of the source and automatically requests server script (if available)
  for formatted HTML. You can also make it show formatted text instead of textarea
  with source markup - if user clicks for editing they'll be swapped, when he blurs
  it'll be restored.

    %%(WikiSample)
      **UverseWiki** - a unified markup language for //all// electronic documents.
    %%

    %%(WikiSample $ajax-wiki.php)   - "$" gets replaced with $settings->mediaURL and a trailing slash

    Supports both %%(WikiSample) //block// and //inline//%% forms.

  This plugin requires common.js and scriptrequest.js. It also needs a server-side
  script to serve formatted HTML - this file can act like one if you put it into
  a public directory and set proper URL; see the end of this file for related comments.

  Possible parameters (each one overrides Settings' if passed):

    * url - URL for AJAX requests; if empty disables editing; if starts with "$" it's
            replaced with $settings->mediaURL (works for $WikiSampleURL too);
    * tip - text that appears under source textarea unless empty;
    * wrap - determines if text will wrap in a multiline source (true by default);
    * editable - an alternative for for specifying empty URL (if set to 0);
    * formatted - shows formatted text instead of source markup.

  CONFIGURATION - fields added to UWikiSettings:

    * $WikiSampleURL = default value for 'url' parameter;
    * $WikiSampleTip = default value for 'tip' parameter;
    * $WikiSampleWrap = default value for 'wrap' parameter;
    * $FormattedWikiSample = default value for 'formatted' parameter.
*/

class UWikiSample {
  const DefaultMarkup = 'wacko';

  static function JsonString($str) {
    return '"'.addcslashes($str, "\\\"\0..\x1F").'"';
  }
}

if (class_exists('UWikiBaseElement')) {
  class Uwikisample_Root extends UWikiBaseElement {
    public $isFormatter = true;
    public $isAction = false;
    public $htmlClasses = array('wiki-sample');
    public $parameterOrder = array('url');

    static $css, $js;

    function Attachments() {
      $params = $this->GetParams();

      $jsReplaces = array(
        '$REQFAILMSG$' => $this->TranslateJSON('%%wikisample: request failed', 'Could not format your text due to the following error error:\n$'),
        '$REQTIMEOUTMSG$' => $this->TranslateJSON('%%wikisample: request timeout', 'Could not format your text - server didn\'t respond in time ($ seconds).'),
        '$URL$' => $params['url'] ? UWikiSample::JsonString($params['url']) : 'null'
      );

      return array('css' => array('WikiSample' => self::$css),
                   'js'  => array('WikiSample' => strtr(self::$js, $jsReplaces)));
    }

      function TranslateJSON($name, $default) {
        return UWikiSample::JsonString( isset($this->strings[$name]) ? $this->strings[$name] : $default );
      }

    function Parse() {
      parent::Parse();

      $this->children[] = $source = $this->NewElement('Uwikisample_Source');
      $source->SetRaw($this->raw);

      $this->children[] = $this->NewElement('Uwikisample_NoScript');

      $formatted = $this->NewElement('Uwikisample_Formatted');

        if ($format = $this->settings->format) {
          $format->root = $formatted;

          if ($format->IsEmpty()) {
            $markup = $this->DefaultStyle(UWikiSample::DefaultMarkup);
            $format->AddFormat($markup);
          } else {
            $markup = $format->NextMarkup();
          }
          $markup === UWikiSample::DefaultMarkup or $this->htmlAttributes['wsmarkup'] = $markup;
        } elseif ($doc = UWikiDocument::TryParsing($this->raw, $this->settings, $markup)) {
          $formatted->children[] = $doc->root;
        } else {
          $formatted = null;
        }

      $formatted and $this->children[] = $formatted;

      $params = $this->GetParams($format);
      $isBlock = $format ? $format->blockExpected : true;

        $this->htmlClasses[] = $params['formatted'] ? 'ws-formatted' : 'ws-source';

        if (isset($params['editable']) and !$params['editable']) { $params['url'] = ''; }

        if (!$params['isDefaultURL'] and $url = &$params['url']) {
          $this->htmlAttributes['wsurl'] = $url;
        }

        if ($params['tip']) {
          $tip = $this->NewElement('Uwikisample_Tip');
          $tip->defaultMessage = $params['tip'];
          array_splice($this->children, 2, 0, array($tip));
        }

        if (!$params['wrap'] and $isBlock) { $source->htmlAttributes['wrap'] = 'off'; }
        if (!$params['url']) { $source->htmlAttributes['readonly'] = 'readonly'; }

      $this->htmlClasses[] = $this->isBlock = $isBlock ? 'ws-block' : 'ws-inline';
      foreach ($this->children as $child) { $child->isBlock = $isBlock; }

      $this->htmlTag = $formatted->htmlTag = $isBlock ? 'div' : 'span';
    }

      function GetParams(UWikiFormat $format = null) {
        $params = $format ? $format->current['params'] : array();

          $params += array('url' => &$this->settings->WikiSampleURL,
                           'tip' => &$this->settings->WikiSampleTip,
                           'wrap' => !isset($this->settings->WikiSampleWrap)
                                     or $this->settings->WikiSampleWrap,
                           'formatted' => !empty($this->settings->FormattedWikiSample));

          $params['isDefaultURL'] = $params['url'] === @$this->settings->WikiSampleURL;

          if (substr($params['url'], 0, 1) === '$') {
            $params['url'] = $this->settings->mediaURL.ltrim($params['url'], '$');
          }

        return $params;
      }
  }

  class Uwikisample_NoScript extends UWikiStringMessage {
    public $htmlTag = 'noscript';
    public $message = '%%wikisample: js disabled';
    public $defaultMessage = 'With JavaScript you can edit this snippet and see your changes on the fly.';
  }

  class Uwikisample_Tip extends UWikiStringMessage {
    public $htmlClasses = array('ws-tip');
    public $message = "\0";
    public $defaultMessage = null;
  }

  class Uwikisample_Source extends Upre_Root {
    public $htmlClasses = array('ws-source');
    public $htmlAttributes = array('onfocus' => 'HookWikiSample(this);');

    function SelfToHtmlWith($contents) {
      if ($this->isBlock) {
        $this->htmlTag = 'textarea';
        $rows = max(2, mb_substr_count($this->raw, "\n") + 1);
        $this->htmlAttributes += array('rows' => $rows, 'cols' => 5);
      } else {
        $this->htmlTag = 'input';
        $this->isSingleHtmlTag = true;
        $this->htmlAttributes['type'] = 'text';
      }

      return parent::SelfToHtmlWith($contents);
    }
  }

  class Uwikisample_Formatted extends UWikiBaseElement {
    public $isBlock = true;
    public $htmlClasses = array('ws-formatted');

    function SelfToHtmlWith($contents) {
      return parent::SelfToHtmlWith("<div>$contents</div>");
    }
  }

Uwikisample_Root::$css = <<<CSS
.wiki-sample { position: relative; }
.wiki-sample.ws-source > .ws-formatted { display: none; }
.wiki-sample > textarea { width: 100%; }
.wiki-sample.ws-formatted > .ws-source { display: none; }
body.ws-edited .ws-tip { display: none; }

.wiki-sample.ws-source > noscript {
  display: none;
  position: absolute;
  z-index: 2;
  margin: -1em 0 0 1em;
  font-style: italic;
  color: #D00;
  border: 1px solid InactiveBorder;
  background: InfoBackground;
  padding: 0.4em;
}

.wiki-sample.ws-block:hover > *, .wiki-sample.ws-block.ws-focused > * { display: block; }
.wiki-sample.ws-inline:hover > *, .wiki-sample.ws-inline.ws-focused > * { display: inline; }
.wiki-sample > .ws-formatted {
  position: absolute;
  z-index: 1;
  border: 1px outset ActiveBorder;
  width: 100%;
  opacity: 1;
  filter: alpha(opacity=100);
}
.wiki-sample > .ws-formatted > div { padding: 0 1em; background: no-repeat left top Window; }
.wiki-sample.ws-block > .ws-formatted { margin-top: 0.5em; }
.wiki-sample.ws-inline > .ws-formatted { margin-left: 0.5em; }
.wiki-sample.ws-busy > .ws-formatted > div {
  background-image: url("data:image/gif;base64,R0lGODlhEAAQAKIHADZmvyRl1FZ5upOjxHWOv7G5yb2/w////yH/C05FVFNDQVBFMi4wAwEAAAAh+QQFAAAHACwAAAAAEAAQAAADQ3i6B8CQORdXCG0eIeC92cZ11seMZBlxjGFUC0EcrgvLcv1W+GzDB1lrxxgMILqi8bhIFgqHJbP5ej6j04gVClxcIwkAIfkEBQAABwAsAAAAABAAEAAAAz94uifCkDkXFwBtHkLgvdnGddbHjGQZcUwQVMswHK4Ly3L9VvhswwcZcFEoDIlFI8xgOCSVESbTCY1Kj4ppJAEAIfkEBQAABwAsAAAAABAADgAAAzt4ukfEkDkXlxBtnjHgvdnGddbHjGQZcQwAVEtRHK4Ly3L9VvhswwcZIxCIGAwQIpFxPA6VzGayCHEqEgAh+QQFAAAHACwAAAAAEAAQAAADPni6N8OQORcXIW2eUuC92cZ11seMZBlxjCBUi2EcrgvLcv1W+GzDBxkDAAAOiUXjAVkMBIzEg9OplE6r1koCACH5BAUAAAcALAAAAAAOABAAAAM8eLpXxVA5F88YbR5j1r3ZxnXWx4xkGXEKQVSM68KtTNc3IwhRECy7HcPnUwR5AMCB+DMik8piBKq8JSEJACH5BAUAAAcALAAAAAAQABAAAAM+eLpnxpA5F1cpbdZzb95cBzLeeAzDGAQnmlbr6r5RzKIquxBEBAAQHo/x+zGEPYHgUAQek8qlcRNdmg7KSgIAIfkEBQAABwAsAAACABAADgAAAz54aqZ+IbzD2Ivx1eaw1Nz1KUUxTQBwlOWppClrurDauq/qDMMpCBMe7/H7PYQ9AuFQBB6TyqURF13iHkpXAgAh+QQFAAAHACwAAAAAEAAQAAADPni6F8GQORfjfADURXPejKeBy7cYBikIB4pu6+qmVcy+4MoURUQQEB6P8fvthIfB4FAEHpPKpXETXZIUykoCADs=");
  opacity: 0.85;
  filter: alpha(opacity=85);
}
CSS;

$defaultMarkup = UWikiSample::DefaultMarkup;
Uwikisample_Root::$js = <<<JAVASCRIPT
var DefaultWikiSampleURL = \$URL$;
var WikiSampleCache = {};

function HookWikiSample(src) {
  if (DebugAssert(src)) {
    var cont = src.parentNode;
    var formatted = FindChildOf(cont, 'ws-formatted');

    var url = cont.getAttribute('wsurl');
    url || (url = DefaultWikiSampleURL);

    if (DebugAssert(cont && formatted && url)) {
      url = url.replace(/[?&]$/, '');
      url += url.indexOf('?') == -1 ? '?callback=' : '&';

      var isInline = HasClass(cont, 'inline') ? 1 : 0;

      function Cache(source, html) {
        isset(html) && (WikiSampleCache[isInline + source] = html);
        return WikiSampleCache[isInline + source];
      }

      var currentReq;
      function Update() {
        if (currentReq) {
          currentReq.End();
          currentReq = null;
        }

        var source = src.value;

        if (isset( Cache(source) )) {
          Response( Cache(source) );
          ReqEnded();
        } else {
          currentReq = new ScriptRequest(url, function (html) {
            Cache(source, html);
            Response(html);
          });

          var markup = cont.getAttribute('wsmarkup') || '$defaultMarkup';
          currentReq.post_data = 'markup=' + markup +
                                 '&inline=' + isInline +
                                 '&source=' + encodeURIComponent(src.value);
          currentReq.on_end = ReqEnded;
          currentReq.Perform();

          ModifyClassNameOf(cont, 'ws-busy');
        }
      }

        function Response(html) {
          if (html != '') { formatted.innerHTML = '<div>' + html + '</div>'; }
          return true;
        }

        function ReqEnded() {
          ModifyClassNameOf(cont, 'ws-busy', false);
        }

        function ReqFailed(reason) {
          if (typeof reason == 'string') {
            var msg = \$REQFAILMSG$.replace('$', reason);
          } else {
            var msg = \$REQTIMEOUTMSG$.replace('$', this.timeout);
          }

          alert(msg);
        }

      var timeout;
      src.onkeyup = src.onchange = function () {
        ModifyClassNameOf(document.body, 'ws-edited');

        clearTimeout(timeout);
        timeout = setTimeout(Update, 35);
      }

      Cache(src.value, formatted.innerHTML);
    }

    src.onfocus = function () { ModifyClassNameOf(cont, 'ws-focused'); }
    src.onblur = function () { ModifyClassNameOf(cont, 'ws-focused', false); }
    src.onfocus();
  }
}
JAVASCRIPT;
}

/*
  Existing RPC parameters are:
  * source = text to format.
  * callback = name of JavaScript function to call back.

  Optional ones:
  * markup = wacko, php, etc.
  * inline = 1/0.
*/

if (count(get_included_files()) < 2 or defined('ServeWikiSamples')) {
  set_time_limit(5);
  ignore_user_abort(false);
  ob_start('ob_gzhandler');
  mb_internal_encoding('UTF-8');
  header('Content-Type: application/javascript; charset=utf-8');

  $_REQUEST += array('markup' => UWikiSample::DefaultMarkup, 'inline' => false);

  if (!class_exists('UWikiDocument')) {
    !defined('UWikiRootPath') and is_file('uversewiki.php') and define('UWikiRootPath', rtrim(getcwd(), '/'));
    !defined('UWikiRootPath') and is_file('../uversewiki.php') and define('UWikiRootPath', rtrim(dirname(getcwd()), '/'));

    if (defined('UWikiRootPath')) {
      $core = UWikiRootPath.'/uversewiki.php';
      if (is_file($core)) { include_once $core; }
    }
  }

  if (!class_exists('UWikiDocument')) {
    echo 'throw "Cannot load UverseWiki framework.";';
  } elseif (!isset($_REQUEST['source']) or !isset($_REQUEST['callback'])) {
    echo 'throw "Wrong request parameters: source or callback or both are missing.";';
  } else {
    $html = null;
    $source = $_REQUEST['source'];

      $useXCache = function_exists('xcache_get');
      $cacheKey = 'WikiSample '.(empty($_REQUEST['inline']) ? 0 : 1).$_REQUEST['markup'].UWikiRootPath;

      if ($useXCache) {
        $cached = xcache_get($cacheKey);
        is_array($cached) and $html = &$cached[$source];
      }

      if (!isset($html)) {
        $doc = new UWikiDocument($source);
        $doc->LoadMarkup($_REQUEST['markup']);
        $doc->settings->LoadFrom(UWikiRootPath.'/config');
        $doc->Parse();

        $doc->BeginRenderingInto('html');
        $html = $_REQUEST['inline'] ? $doc->root->InlineHTML() : $doc->ToHTML();
        $doc->EndRenderingInto('html', $html);

        if ($useXCache) {
          $cached[$source] = &$html;
          xcache_set($cacheKey, $cache);
        }
      }

    echo $_REQUEST['callback'].'('.UWikiSample::JsonString($html).');';
  }
}
