<?php

class EMail {
  static $mimeVersion = 1.0;
  static $bodyTypes = array('text/plain' => 'text', 'text/html' => 'html');
  static $defaultMIME = 'application/octet-stream';
  static $boundaryPrefix = '_hello-uverse_';
  static $mimeStub = 'Your e-mail client does not support multi-part MIME messages.';

  static $mimeByExt = array(
    'jpeg' => 'image/jpeg', 'jpg' => 'image/jpeg', 'png' => 'image/png', 'gif' => 'image/gif',
    'bmp' => 'image/x-ms-bmp', 'svg' => 'image/svg+xml', 'ico' => 'image/vnd.microsoft.icon',
    'txt' => 'text/plain', 'log' => 'text/plain',  'wiki' => 'text/plain',
    'wacko' => 'text/plain', 'ini' => 'text/plain', 'sh' => 'text/plain',
    'conf' => 'text/plain', 'pas' => 'text/x-pascal', 'c' => 'text/x-c',
    'h' => 'text/x-c', 'cpp' => 'text/x-c', 'hpp' => 'text/x-c',
    'pdf' => 'application/pdf', 'doc' => 'application/msword',
    'docx' => 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
    'xls' => 'application/msexcel',
    'xlsx application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
    'rtf' => 'application/rtf', 'tar' => 'application/x-tar', 'gz' => 'application/x-gzip',
    'bz2' => 'application/x-bzip2', 'rar' => 'application/x-rar-compressed',
    'zip' => 'application/zip', '7z' => 'application/x-7z-compressed',
    'htm' => 'text/html', 'html' => 'text/html', 'php' => 'text/html',
    'xml' => 'application/xhtml+xml', 'mht' => 'message/rfc822',
    'swf' => 'application/x-shockwave-flash', 'css' => 'text/css',
    'dtd' => 'application/xml-dtd', 'cfg' => 'text/plain', 'manifest' => 'text/plain',
    'exe' => 'application/x-msdownload', 'dll' => 'application/x-msdownload');

  // null fields here and below use BConfig values.
  public $from, $returnPath, $subject;
  // to/copyTo/bccTo addressee is of form [name ]e@mail OR name <e@mail> - in the first
  // form, if name is followed by a space it's converted to the second (full) form.
  public $to = array(), $copyTo = array(), $bccTo = array();
  public $deliveryNotifications = array();    // array of e-mails.
  public $headers = array();  // override standard (From, Content-Type, etc.) headers.
  public $allowedHeaders;     // see BConfig::$mail['allowedHeaders'].

  public $bodyEncoding, $bodyCharsets, $forceBodyCharset;
  public $makeTextBodyFromHTML, $allowedBodyMIME, $textBodyFormat;
  public $allowedAttachments;
  public $params;               // custom parameters for sendmail (last arg of PHP mail()).

  public $headerEOLN, $sortHeaders;
  public $eoln = "\r\n";
  public $skipRelatedAttIfNoHtmlBody = true;

  // set after Send(); array with keys: subject, headers, message,
  // status (return value of mail() or null if it wasn't called, e.g. when simulating).
  public $mailed;
  public $simulateSending;      // is initially set by Send() and SimulateSending().

  protected $body = array();    // keys 'text', 'html'.
  // array of arrays with keys: name, mime (Content-Type), data, headers,
  // isRelated (puts in multipart/related group).
  protected $attachments = array();
  // 0 (low), 1 (normal), 2 (high) - X-Priority and Importance headers.
  protected $priority = 1;

  static function ChunkBase64($str) { return chunk_split(base64_encode($str)); }

  static function MangleUTF8($str) {
    if (!isset($str[0]) or utf8_decode($str) === $str) {
      return $str;
    } else {
      return '=?UTF-8?B?'.base64_encode($str).'?=';
    }
  }

  static function DemangleUTF8($str) {
    static $pf = '=?UTF-8?B?';

    if (substr($str, -2) === '?=' and substr($str, 0, strlen($pf)) === $pf) {
      return base64_decode( substr($str, strlen($pf), -2) );
    } else {
      return $str;
    }
  }

  static function OneLiner($str) {
    for ($i = 0; isset($str[$i]); ++$i) { ord($str[$i]) < 32 and $str[$i] = ' '; }
    return $str;
  }

  static function NormAddress($addr) {
    $add = self::OneLiner($addr);
    $delim = mb_strpos($addr, '<');

      $delim === false and $delim = mb_strrpos($addr, ' ');
      $delim === false and $delim = -1;

      $name = $delim == -1 ? '' : trim( mb_substr($addr, 0, $delim) );
      $addr = rtrim(trim( mb_substr($addr, $delim + 1) ), '>');

    if (mb_strpos($addr, '@') === false) {
      throw new BException('Wrong e-mail address format.', $addr);
    }

    $name === '' or $name = '"'.self::MangleUTF8($name).'" ';
    return "$name<$addr>";
  }

  static function ExtractFrom($addr) {
    $head = strtok(self::NormAddress($addr), '<');
    $tail = strtok('>');
    return trim($tail === false ? $head : $tail, ' "<>');
  }

  static function SetConfigTo(EMail $email) {
    static $fields = array('from', 'returnPath', 'bodyEncoding', 'bodyCharsets',
                           'forceBodyCharset', 'makeTextBodyFromHTML',
                           'textBodyFormat', 'headerEOLN', 'sortHeaders', 'params',
                           'allowedHeaders');

    foreach ($fields as $field) {
      $email->$field === null and $email->$field = BConfig::$mail[$field];
    }

    if (!isset($email->allowedAttachments) and $att = BConfig::$mail['attachments']) {
      $email->allowedAttachments = $att;
    }

    $email->headers += BConfig::$mail['extraHeaders'];
    isset($email->from) or $email->from = BConfig::$mail['from'];
    isset($email->allowedBodyMIME) or $email->allowedBodyMIME = BConfig::$mail['bodyMIME'];
  }

  static function SetHeadersTo(EMail $email) {
    static $xPriority = array('5 (Lowest)', '3 (Normal)', '1 (Highest)');
    static $importance = array('Low', 'Normal', 'High');

    if (!$email->to and !$email->copyTo and !$email->bccTo) {
      return true;    // No recepients - stop e-mail sending event.
    }

    $priority = $email->Priority();
    $email->headers += array(
      'MIME-Version' => self::$mimeVersion, 'X-Mailer' => 'UverseBlog',
      'Date' => date('r'),
      'X-Priority' => $xPriority[$priority], 'Importance' => $importance[$priority]);

    $to = $email->to;
    array_shift($to);
    $to and $email->headers += array('To' => $email->AddressHeader($to));

    $email->copyTo and $email->headers += array('Cc' => $email->AddressHeader($email->copyTo));
    $email->bccTo and $email->headers += array('Bcc' => $email->AddressHeader($email->bccTo));

    if ($from = $email->from) {
      $email->headers += array('From' => self::NormAddress($from),
                               'Message-ID' => '<'.time().'-'.self::ExtractFrom($from).'>');
    }

    if ($returnPath = $email->returnPath) {
      $email->headers += array('Return-Path' => self::NormAddress($returnPath));
    }

    if ($delivAddrs = &$email->deliveryNotifications) {
      $delivHdr = &$email->headers['Disposition-Notification-To'];

      if (isset($delivHdr)) {
        is_array($delivHdr) or $delivHdr = array($delivHdr);
      } else {
        $delivHdr = array();
      }

      $delivHdr[] = $email->AddressHeader($delivAddrs);
    }
  }

  static function Mail(EMail $email) { $email->DoMail(); }

  static function EncodeBody(&$body, &$type, array &$headers, EMail $email) {
    $charset = $email->ConvertCharsetOf($body);

    $encoding = strtolower($email->bodyEncoding);
    switch ($encoding) {
    case 'base64':
      $body = self::ChunkBase64($body);
      break;

    case 'qp':
    case 'quotedprintable':
      $encoding = 'quoted-printable';
    case 'quoted-printable':
      $body = quoted_printable_encode($body);
      break;

    case 'plain':
      $encoding = '8bit';
      break;
    }

    $format = '';
    $email->textBodyFormat and $format = '; format='.$email->textBodyFormat;

    $mime = $type === 'html' ? 'text/html' : 'text/plain';
    $headers += array('Content-Type' => "$mime; charset=$charset$format",
                      'Content-Transfer-Encoding' => $encoding);
  }

  static function DenyAttachment(&$att, EMail $email) {
    // internal attachments (e.g. mail body) have neither name nor mime.
    if (isset($att['name']) and isset($att['mime'])) {
      $allowed = &$email->allowedAttachments;
      is_array($allowed) or $allowed = $email->ParseAllowedAttachments($allowed);

      $ext = strtolower( ExtOf($att['name']) );
      $mime = strtolower( $att['mime'] );

        foreach ($allowed as $key => $item) {
          if ($key !== '-') {
            $matches = $item[0] === '.' ? ($ext !== '' and MatchWildcard($item, $ext, true))
                                        : ($mime and MatchWildcard($item, $mime, true));

            if ( empty($allowed['-']) ? !$matches : $matches ) {
              $att = null;
              return;
            }
          }
        }
    }
  }

  static function EncodeAttachment(&$att, EMail $email) {
    if ($mime = &$att['mime']) {
      $name = empty($att['isRelated']) ? '' : '; name="'.self::MangleUTF8($att['name'], true).'"';
      $att['headers'] += array('Content-Type' => $mime.$name);

      $encHeader = &$att['headers']['Content-Transfer-Encoding'];
      if (empty($encHeader)) {
        if (strtok($mime, '/') === 'text' and strtolower($email->bodyEncoding) !== 'base64') {
          $encHeader = 'quoted-printable';
          $att['data'] = quoted_printable_encode($att['data']);
        } else {
          $encHeader = 'base64';
          $att['data'] = self::ChunkBase64($att['data']);
        }
      }
    }

    if (empty( $att['isRelated'] )) {
      if (isset($att['name'])) {
        $disp = 'attachment; filename="'.self::MangleUTF8($att['name'], true).'"';
        $att['headers'] += array('Content-Disposition' => $disp);
      }
    } else {
      $att['headers'] += array('Content-ID' => '<'.self::MangleUTF8($att['name']).'>');
    }
  }

  static function Transmit(&$subject, &$headers, &$body, EMail $email) {
    if (!$email->simulateSending) {
      if ($email->to) {
        $to = $email->to[0];
      } else {
        $to = $email->from;
      }

      $allowed = $email->allowedHeaders;
      if ($allowed and empty($allowed['-'])) {
        $headers = array_intersect_key($headers, $allowed);
      } else {
        $headers = array_diff_key($headers, $allowed);
      }

      $email->mailed['status'] =
        mail(self::NormAddress($to), self::OneLiner($subject),
             $body, $email->JoinHeaders($headers), $email->params);
    }
  }

  // $att defines attachments: true - Template::AttachmentsOf, false/null - none, array() - set given.
  static function SendFromTemplate($recepient, $tpl, array $vars, $subject = null, $att = true) {
    $tpl = "email - $tpl";
    $vars += EMailVars::CommonFor($recepient);
    $texts = Template::FormatWrapping($tpl, $vars);

    if ($texts) {
      if ($subject === null) {
        $subject = Template::LiteFormat(Translate("$tpl: subject"), $vars);
      }

      $obj = new EMail($recepient, $subject, $texts);

        if ($att === true) {
          $obj->Attachments(Template::AttachmentsOf($tpl));
        } elseif (is_array($att)) {
          $obj->Attachments($att);
        }

      return $obj->Send();
    }
  }

  function __construct($to, $subject, array $body = array()) {
    $this->to[] = $to;
    $this->subject = $subject;
    $this->Body($body);
  }

  function Priority($level = null) {
    if ($level !== null) {
      if (!is_numeric($level) or $level < 0 or $level > 2) {
        throw new BException('Invalid Priority($level) value - 0-2 expected.', "$level given");
      } else {
        $this->prioerity = (int) $level;
      }
    }

    return $this->priority;
  }

  // get all:     function ()
  // get by MIME: function ('html')
  // set by MIME: function ('html', '<html>...')
  // set all:     function (array('html' => '<html>...'))
  function Body($newOrType = array(), $body = null) {
    if (is_array($newOrType)) {
      $norm = array();

        foreach ($newOrType as $type => &$item) {
          if (!is_string($item) and !is_scalar($item)) {
            throw new BException("Wrong \$new[$type] for EMail->Body().",
                                 var_export($item, true));
          }

          $type = strtolower($type);
          isset(self::$bodyTypes[$type]) and $type = self::$bodyTypes[$type];
          $norm[$type] = &$item;
        }

      $this->body = $norm;
    } else {
      isset(self::$bodyTypes[$newOrType]) and $type = self::$bodyTypes[$newOrType];
      $body === null or $this->body[$newOrType] = $body;
    }

    if (is_string($newOrType)) {
      return $this->body[$newOrType];
    } else {
      return $this->body;
    }
  }

  function Attach($name, $data, $mime = null, $headers = array(), $isRelated = false) {
    $mime === null and $mime = $this->MimeByExt($name);
    $mime = strtolower($mime);

    $this->attachments[$name] = compact('name', 'mime', 'data', 'headers', 'isRelated');
  }

    function AttachRelated($name, $data, $mime = null, $headers = array()) {
      Attach($name, $data, $mime, $headers, true);
    }

  // $field = name (str), mime (str), data (str), headers (array).
  function Attachment($name, $field = 'data') {
    if (isset($this->attachments[$name])) {
      return $field ? $this->attachments[$name][$field] : $this->attachments[$name];
    }
  }

  function Attachments($new = null) {
    if (is_array($new)) {
      $this->ClearAttachments();

      foreach ($new as $name => $att) {
        is_array($att) or $att = array('data' => $att);
        $att += array('name' => $name, 'mime' => null, 'headers' => array(), 'isRelated' => false);

        if (!isset($att['data'])) {
          throw new BException('EMail->Attachments() received an array item without'.
                               ' \'data\' key.', var_export($att, true));
        }

        $this->Attach($att['name'], $att['data'], $att['mime'], $att['headers'], $att['isRelated']);
      }
    }

    return $this->attachments;
  }

  function AttachmentNames() { return array_keys($this->attachments); }
  function ClearAttachments() { $this->attachments = array(); }

  function MimeByExt($ext, $default = true) {
    $ext = strtolower( ltrim(ExtOf($ext), '.') );
    $default = $default ? self::$defaultMIME : $ext;
    return isset(self::$mimeByExt[$ext]) ? self::$mimeByExt[$ext] : $default;
  }

  // after Send() is called this object's props will be modified depending on
  // the settings (body might be reconverted, attachments removed, etc.).
  function Send() {
    $this->Fire(false);
    return $this->mailed['status'];
  }

    function SimulateSending() {
      return $this->Fire(true);
    }

    protected function Fire($simulate) {
      $this->simulateSending = $simulate;
      $this->mailed = array('subject' => null, 'headers' => null,
                            'message' => null, 'status' => null);

      BEvent::Fire('mail: send', array($this));
      return $this->mailed;
    }

  function AddressHeader(array $addresses) {
    foreach ($addresses as &$addr) { $addr = self::NormAddress($addr); }
    return join(', ', $addresses);
  }

  function DoMail() {   // called from 'mail: send' event.
    $subject = self::MangleUTF8($this->subject);

    $name = $mime = null;

    $headers = array();
    $body = $this->PrepareBody($this->body);
    $hasHtmlBody = isset($body['html']);
    $data = $this->BuildBody($body, $headers);

    $files = $this->attachments;
    $related = array();

      foreach ($files as &$file) {
        if ($file['isRelated']) {
          $related[] = $file;
          $file = null;
        }
      }

    $files = array_filter($files);

      if ($related and (!$this->skipRelatedAttIfNoHtmlBody or $hasHtmlBody)) {
        array_unshift($related, compact('name', 'mime', 'data', 'headers'));

        $headers = array();
        $data = $this->BuildRelatedAttachments($related, $headers);
      }

      if ($files) {
        array_unshift($files, compact('name', 'mime', 'data', 'headers'));

        $headers = array();
        $data = $this->BuildAttachments($files, $headers);
      }

    $headers += $this->headers;

    $this->mailed['subject'] = $subject;
    $this->mailed['headers'] = $headers;
    $this->mailed['message'] = &$data;

    BEvent::Fire('mail: transmit', array(&$subject, &$headers, &$data, $this));
  }

    function &PrepareBody($bodies) {
      $allowed = $this->allowedBodyMIME;

        foreach ($allowed as &$one) { $one = strtolower($one); }
        $allowed = array_flip($allowed);

        foreach (self::$bodyTypes as $mime => $type) {
          isset($allowed[$mime]) and $allowed[$type] = true;
        }

      if ($this->makeTextBodyFromHTML and isset($allowed['html']) and
          !isset($bodies['text']) and isset($bodies['html'])) {
        $bodies['text'] = strip_tags($bodies['html']);
      }

      if (!empty($allowed)) {
        $bodies = array_intersect_key($bodies, $allowed);
      }

      return $bodies;
    }

    function &BuildBody($bodies, array &$headers) {
      if (!$bodies) { throw new BException('No EMail bodies when sending e-mail.'); }

      $result = '';

      if (count($bodies) > 1) {
        $boundary = $this->GenerateMimeBoundaryFor($bodies);
        $headers += array('Content-Type' => 'multipart/alternative; boundary="'.$boundary.'"');
        $result .= self::$mimeStub.$this->eoln;
      } else {
        $boundary = null;
      }

      $footers = null;

        foreach ($bodies as $type => &$body) {
          is_int($type) and $type = null;   // internal - message body itself (multipart/alternative or related).

          $bodyHeaders = array();
          $args = array(&$body, &$type, &$bodyHeaders, $this);
          BEvent::Fire('mail: prepare body', $args);

          if ($type) {
            if (!isset($footers)) {
              $footers = BEvent::FireResult('array', 'mail: footer', array($bodies, $this));
            }

            $this->SetFootersTo($body, $type, $footers);
          }

          BEvent::Fire('mail: encode body', $args);

          if (isset($boundary)) {
            $body = $this->JoinHeaders($bodyHeaders, true).$body;
            $result .= "--$boundary{$this->eoln}$body{$this->eoln}";
          } else {
            $headers += $bodyHeaders;
            $result = &$body;
          }
        }

      isset($boundary) and $result .= "--$boundary--".$this->eoln;
      return $result;
    }

      function SetFootersTo(&$body, $type, array $footers) {
        $useHTML = ($type === 'html' and $pos = mb_stripos($body, '</body>'));

        if ($useHTML) {
          $prefix = "{$this->eoln}      <div class=\"%s\">{$this->eoln}        ";
          $suffix = "{$this->eoln}      </div>";
        } else {
          $prefix = Translate('email - footer prefix');
          $suffix = Translate('email - footer suffix');
        }

        $joined = '';

          foreach ($footers as $footer) {
            $footer = &$footer[$type];

            if (isset($footer)) {
              $footer = (array) $footer;
              $footer += array('', '');
              list($text, $classes) = $footer;

              $classes = "$classes" === '' ? 'footer' : "footer $classes";
              $pf = $useHTML ? sprintf($prefix, $classes) : $prefix;
              $joined .= $pf.$text.$suffix;
            }
          }

        if ($joined !== '') {
          if ($useHTML) {
            $joined = "{$this->eoln}    <div class=\"footers\">$joined{$this->eoln}    </div>{$this->eoln}  ";
            $body = mb_substr($body, 0, $pos). $joined .mb_substr($body, $pos);
          } else {
            $body .= $joined;
          }
        }
      }

    function BuildAttachments(array $files, array &$headers, $mime = 'multipart/mixed') {
      $boundary = $this->GenerateMimeBoundaryFor($files);
      $headers += array('Content-Type' => "$mime; boundary=\"$boundary\"");

      $result = '';

        foreach ($files as &$file) {
          BEvent::Fire('mail: encode attachment', array(&$file, $this));
          if (is_array($file)) {
            $result .= "--$boundary{$this->eoln}".
                       $this->JoinHeaders($file['headers'], true).$file['data'].$this->eoln;
          }
        }

    $result .= "--$boundary--".$this->eoln;
    return $result;
  }

    function BuildRelatedAttachments(array $files, array &$headers) {
      return $this->BuildAttachments($files, $headers, 'multipart/related');
    }

  // $bodies can contain strings or arrays containing 'data' keys (useful for attachments).
  function GenerateMimeBoundaryFor(array &$bodies) {
    for ($tries = 0; $tries < 10; ++$tries) {
      $boundary = uniqid(self::$boundaryPrefix, $tries > 0);

      $found = false;

        foreach ($bodies as &$s) {
          $found |= strpos(is_array($s) ? $s['data'] : $s, $boundary) !== false;
          if ($found) { break; }
        }

      if (!$found) { return $boundary; }
    }

    throw new BException('Too many tries made to generate e-mail MIME boundary.');
  }

  function JoinHeaders(array $headers, $doubleEOLN = false) {
    $res = '';

    $this->sortHeaders and ksort($headers);

    foreach ($headers as $name => $value) {
      $value = (array) $value;
      foreach ($value as $item) { $res .= "$name: $item{$this->headerEOLN}"; }
    }

    return $res.($doubleEOLN ? $this->headerEOLN : '');
  }

  function ParseAllowedAttachments($allowed) {
    switch ($allowed) {
    case '':  case '-':   case '0':
      return array('-' => true);

    case '1':
      return array('-' => false);

    default:
      $deny = $allowed[0] === '-';
      $allowed = array('-' => $deny);

      foreach (explode(' ', strtolower( ltrim($allowed, '-') )) as $item) {
        $item === '' or $allowed[$item] = true;
      }

      return $allowed;
    }
  }

  function ConvertCharsetOf(&$body) {
    $charset = 'utf-8';

    foreach ($this->bodyCharsets as $newCharset) {
      $converted = BConfig::ConvertCharset($newCharset, false, $body);

      $doConvert = ($this->forceBodyCharset or
                    $body === BConfig::ConvertCharset($newCharset, true, $converted));

      if ($doConvert) {
        $body = $converted;
        $charset = $newCharset;
        break;
      }
    }

    return $charset;
  }
}

if (!function_exists('quoted_printable_encode')) {
  // Reformatted implementation of quoted_printable_encode() for PHP < 5.3 taken from:
  // http://www.php.net/manual/en/function.quoted-printable-encode.php#106078
  function quoted_printable_encode($str, $maxLineLength = 75) {
    $lp = 0;
    $ret = '';
    $hex = '0123456789ABCDEF';
    $length = strlen($str);
    $str_index = 0;

    while ($length--) {
      if ((($c = $str[$str_index++]) == "\015") && ($str[$str_index] == "\012") && $length > 0) {
        $ret .= "\015";
        $ret .= $str[$str_index++];
        $length--;
        $lp = 0;
      } elseif (ctype_cntrl($c) || (ord($c) == 0x7f) || (ord($c) & 0x80) || ($c == '=') ||
                (($c == ' ') && ($str[$str_index] == "\015"))) {
        if (($lp += 3) > $maxLineLength) {
          $ret .= '=';
          $ret .= "\015";
          $ret .= "\012";
          $lp = 3;
        }

        $ret .= '=';
        $ret .= $hex[ord($c) >> 4];
        $ret .= $hex[ord($c) & 0xf];
      } else {
        if ((++$lp) > $maxLineLength) {
          $ret .= '=';
          $ret .= "\015";
          $ret .= "\012";
          $lp = 1;
        }

        $ret .= $c;
      }
    }

    return $ret;
  }
}
