<?php

class PluginInstallator {
  public $mode = 'install';
  public $file;
  public $deleteSetupFile = false;

  public $name, $prefaceSuffix = array();
  public $files = array();
  // extra info. Standard fields are: 'on uninstall', 'on finish' (called after install is
  // complete), 'user files'.
  public $info = array();
  // variables set during setup file parsing:
  public $installTo, $sectionOptions;

  // Note: uninstallation process must continue even in case of (non-critical) errors so
  // it can be used to clean up all traces of a plugin even if it wasn't installed properly.
  static function Uninstall($name) {
    $name = strtolower($name);

    $func = Plugins::Info($name, 'on uninstall');
    $func and BEvent::CallSingle($func, array($name));

    BEvent::Fire('on plugin uninstall', array(&$name));
  }

    static function Unregister($plugin) {
      $eventFile = self::EventFile();
      if (is_file($eventFile) and filesize($eventFile)) {
        $prefix = preg_quote(self::GetRegPrefixFor($plugin), '~');
        $clean = preg_replace('~\n+\s*'.$prefix.'.*?(\n\s*/\* @|$)~us', '\1',
                               file_get_contents($eventFile));

        if (trim($clean) === '<?php') {
          unlink($eventFile);
          $failed = is_file($eventFile);
        } else {
          $failed = !is_int( file_put_contents($eventFile, $clean, LOCK_EX) );
        }

          if ($failed) {
            throw new BException("Cannot update $eventFile - ensure it's writable.");
          }
      }
    }

      static function EventFile() { return BConfig::$paths['config'].'plugins.php'; }
      static function GetRegPrefixFor($name) { return "/* @$name: */\n"; }

    static function RemoveFilesOf($plugin) {
      $info = BIndex::SelectFrom('plugins', $plugin);
      if ($info['files']) {
        foreach ($info['files'] as $file => $md5) {
          $current = md5_file($file);
          if ($current !== $md5) {
            BEvent::Fire('on different plugin checksum', array($plugin, $file, (string) $current, $md5));
          } else {
            unlink($file);
            RemoveEmptyDirsAbove(dirname($file));
          }
        }
      }
    }

  static function GetRegSectionOf($plugin, $withPrefix = false) {
    $eventFile = self::EventFile();
    if (is_file($eventFile) and filesize($eventFile)) {
      $prefix = preg_quote(self::GetRegPrefixFor($plugin), '~');
      if (preg_match('~\n+(\s*'.$prefix.'\n*)(.*?)\n(\s*/\* @|$)~us',
          file_get_contents($eventFile), $matches)) {
        return trim( ($withPrefix ? $matches[1] : '').$matches[2], "\r\n" );
      }
    }
  }

  function __construct($file) {
    $this->file = $file;
    if (!is_file($file)) { throw new PluginInstError($this, 'its setup file doesn\'t exist'); }
  }

    static function NormalizeFile($installator, &$file) {
      $file = str_replace("\r\n", "\n", $file);
      $file = str_replace('<?=', '<?php echo ', trim($file, "\n"));
    }

    static function Finish($installator) {
      $info = array('files' => $installator->files) + $installator->info;
      BIndex::AddTo('plugins', $installator->name, $info);

      $func = &$installator->info['on finish'];
      $func and BEvent::CallSingle($func, array($installator));
    }

  function Run() {
    BEvent::Fire('on plugin install', array($this));
    require $this->file;

      if (!$this->name) { throw new PluginInstError($this, 'it didn\'t set its own name'); }

      if (!preg_match('/^[a-z0-9 _]+$/', $this->name)) {
        throw new PluginInstError($this, 'plugin name can contain only a-z, 0-9, _ and '.
                                 ' space characters but "'.$this->name.'" was given.');
      }

      $this->name = strtolower($this->name);

    if ($this->mode !== 'debug') {
      $this->Install();
      BEvent::Fire('plugin installed', array($this));

      if ($this->deleteSetupFile) {
        unlink($this->file);
        if (is_file($this->file)) {
          throw new PluginInstError($this, 'cannot delete its setup file after'.
                                    ' successful installation');
        }
      }
    }
  }

  function Install() {
    BEvent::Fire('installing plugin', array($this));

    $script = file_get_contents($this->file);
    BEvent::Fire('prepare plugin file', array($this, &$script));

    $pieces = preg_split('~\n */\* *(install:? +.+|one-shot|options.*) *\*/ *(?=\r?\n)~iu',
                         $script, -1, PREG_SPLIT_DELIM_CAPTURE);
    if (($e = preg_last_error()) !== 0) {
      throw new PluginInstError($this, "preg_last_error() returned non-zero code ($e) - check".
                                ' that its setup file is encoded in UTF-8');
    }

    foreach ($pieces as $i => &$piece) {
      if ($i % 2 === 1) {
        $piece = trim($piece);
        BEvent::Fire('on plugin directive', array($this, &$piece));
      } elseif ($i === 0) {
        $this->Preface($piece);
      } else {
        BEvent::Fire('on plugin section', array($this, &$piece));
      }
    }
  }

  function Preface($piece) {
    BEvent::Fire('on plugin preface', array($this, &$piece));

    if (substr($piece, 0, 5) !== '<?php') {
      throw new PluginInstError($this, 'setup script must begin with "<?php"');
    }

    $this->mode === 'overwrite' and self::Unregister($this->name);

    $piece = trim(substr($piece, 5), "\r\n");
    $eventFile = self::EventFile();
    $prefix = self::GetRegPrefixFor($this->name);

    if (is_file($eventFile) and filesize($eventFile)) {
      if (strpos(file_get_contents($eventFile), $prefix) !== false) {
        throw new PluginInstError($this, 'it\'s already registered (uninstall it first or use overwrite mode)');
      }
    } else {
      $written = file_put_contents($eventFile, "<?php\n");
      if (!is_int($written)) {
        throw new PluginInstError($this, "cannot register it in $eventFile - ensure it's writable");
      }
    }

    // todo: move writing (any) file to a method that'll check for is_int($bytesWritten)
    //       and raise exception otherwise.
    $piece = "\n  $prefix$piece\n".join("\n", $this->prefaceSuffix)."\n";
    file_put_contents($eventFile, $piece, FILE_APPEND);
  }

  static function Directive($installator, $name) {
    if ($name[0] === 'o') {     // one-shot, options; piece is used on install-time only.
      $installator->installTo = null;
    } else {
      $installTo = trim( substr($name, strlen('install')), ': ' );

      if (strpos($installTo, ':')) {
        $options = explode(',', strtok($installTo, ':'));
        foreach ($options as &$opt) { $opt = trim($opt); }
        $installator->sectionOptions = array_flip($options);

        $installTo = ltrim( strtok(null) );
      } else {
        $installator->sectionOptions = array();
      }

      if (ExtOf($installTo) === '') {
        $installator->installTo = BConfig::$paths['plugins']."$installTo.php";
      } else {
        $installator->installTo = BConfig::$enginePath.$installTo;
      }
    }
  }

  static function Base64Section($installator, &$code) {
    if (isset( $installator->sectionOptions['base64'] )) {
      $code = base64_decode($code);
      if (!is_string($code) or strlen($code) == 0) {
        throw new PluginInstError($installator, "its component file $installTo is expected to be a valid base64 stream but its decoding failed");
      }
    }
  }

  static function OverwriteSection($installator, &$code) {
    $installTo = &$installator->installTo;
    if (!$installTo) { return; }

    foreach ($installator->sectionOptions as $opt) {
      if (strpos($opt, 'overwrite') === 0) {
        $hashes = explode(' ', $opt);
        array_shift($hashes);
      }
    }

    if (isset($hashes)) {
      foreach ($hashes as &$hash) {
        $hash = trim($hash);
        if ($hash !== '' and strlen($hash) !== 32) {
          throw new BException('Overwrite option of /*Install*/ directive met a hash of wrong'.
                               ' length (32 expected, '.strlen($hash).' got: '.$hash.')');
        }
      }

      $hashes = array_filter($hashes);
      if (!$hashes) {
        throw new BException('Overwrite option of /*Install*/ directive needs at least one hash.)');
      }

      $gotHash = md5_file($installTo);
      foreach ($hashes as &$hash) {
        if ($gotHash === $hash) {
          unlink($installTo);
          clearstatcache();
          return;
        }
      }

      // null marks section as install-time and no writing happens (see Section).
      $installTo = null;
    }
  }

  static function UnlessSectionExists($installator, &$code) {
    $installTo = &$installator->installTo;
    if ($installTo and is_file($installTo)) {
      $installTo = null;
    }
  }

  static function Section($installator, $code) {
    $installTo = $installator->installTo;
    if ($installTo !== null) {
      if (isset($code[0]) and $code[0] === "\n") {
        // directive comment usually has a line break: /* install */ \n < ?php...
        $code = substr($code, 1);
      }
      if (ExtOf($installTo) === '.php' and strpos($code, '<?php') === false) {
        $code = "<?php\n$code";
      }

      if (is_file($installTo) and trim(file_get_contents($installTo)) !== trim($code)
          and $installator->mode !== 'overwrite') {
        throw new PluginInstError($installator, "its component file $installTo already exists and is different");
      } else {
        MkDirOf($installTo);
        if (!is_int( file_put_contents($installTo, $code) )) {
          throw new PluginInstError($installator, "cannot write $installTo");
        }

        $installator->files[$installTo] = md5($code);
      }
    }
  }

  /* Methods useful for calling from a setup file loaded in Run(): */
  function Caption($str, $lang = '') {
    if (!is_string($str)) {
      throw new PluginInstError($this, 'Caption() accepts only strings but '.gettype($str).' was given.');
    }

    $lang and $lang = " $lang";
    $this->prefaceSuffix['caption'.$lang] =
      "  BConfig::\$strings['{$this->name}: plugin caption$lang'] = ".var_export($str, true).';';
  }
}

class PluginInstError extends BException {
  public $installator;

  function __construct($installator, $msg) {
    $name = $installator->name ? $installator->name : basename($installator->file, '.php');
    $msg = 'Failed to '.$installator->mode.' '.$name.': '.$msg;
    parent::__construct($msg, 'setup file: '.$installator->file);

    $this->installator = $installator;
  }
}
