<?php

class Sidebars {
  static $datePartsRegExp = '~^(?P<year>\d\d(\d\d)?)(\.(?P<month>\d\d?)(\.(?P<day>\d\d?)(\s+(?P<hour>\d\d?)(:(?P<minute>\d\d?)(:(?P<second>\d\d?))?)?)?)?)?$~';

  static function GetAll(array &$result) {
    $files = glob(BConfig::$paths['config'].'sidebars/*.{conf,html,wiki}', GLOB_BRACE);
    $info = self::InfoFrom($files);

    foreach ($info as $sidebar) {
      if (empty($sidebar['type'])) {
        throw new BException("No 'type' set for a sidebar.", var_export($sidebar['file'], true));
      }

      BEvent::Fire('sidebars: generate', array(&$sidebar));

      $cont = &$sidebar['content'];
      if ( is_array($cont) ? $cont : ("$cont" !== '') ) {
        $result[] = $sidebar;
      }
    }
  }

    static function &InfoFrom(array $files) {
      static $extToType = array('.conf' => 'plugin', '.html' => 'html', '.wiki' => 'wiki');
      $info = array();

        foreach ($files as $file) {
          $data = file_get_contents($file);
          $file = BConfig::ToUTF8('file name', $file);

          $ext = ExtOf($file);
          $id = basename($file);

          $type = $extToType[$ext];
          if (!$type) {
            throw new BException('Unexpected sidebar file extension.', "extension: $ext");
          }

          $info[$file] = compact('file', 'ext', 'type', 'id')
                         + BEvent::FireResult('array', 'sidebars: parse', $data)
                         + BEvent::FireResult('array', 'sidebars: find files', $file);
        }

      return $info;
    }

      static function Parse(array &$result, $str) {
        list($config, $caption) = explode("\n==", "\n$str", 2);
        @list($caption, $rawContent) = explode("\n", $caption, 2);

        $config = ParseKeyValuesExt($config);

        $caption = trim($caption);
        if ($caption === '') {
          $caption = null;
        } elseif (mb_substr($caption, -2) === '==') {
          $caption = trim(mb_substr($caption, 0, -2));
        }

        $result = compact('config', 'caption', 'rawContent');
      }

      static function FindFiles(array &$result, $file) {
        static $sizes = '16,24,32,48,64';

        $file = BConfig::FromUTF8('file name', $file);

        $file = ChangeStrExt($file, '-{'.$sizes.'}.{png,gif,jpg,jpeg,bmp}');
        $images = glob($file, GLOB_BRACE | GLOB_NOSORT);

        foreach ($images as $file) {
          $file = BConfig::ToUTF8('file name', $file);

          $name = mb_substr($file, 0, -1 * strlen(ExtOf($file)));
          $short = rtrim($name, '0..9');
          $size = mb_substr( $name, -1 * (strlen($name) - strlen($short)) );

          $url = BConfig::$sidebarURL.basename($file);
          $result['icon'.$size] = $url;
        }
      }

  static function HideIfHidden(array &$info) {
    $isShowing = null;

      foreach ($info['config'] as $key => &$value) {
        $show = strpos($key, 'show ') === 0;

        if ($show or strpos($key, 'hide ') === 0) {
          if (isset($isShowing) and $isShowing !== $show) {
            throw new BException('Sidebar has both \'show\' and \'hide\' settings.',
                                 var_export($info, true));
          } else {
            $isShowing = $show;
          }
        }
      }

    if (isset($isShowing)) {
      $keyPrefix = $isShowing ? 'show' : 'hide';
      if (isset($info['config'][$keyPrefix])) {
        $asAND = $info['config'][$keyPrefix] === 'and';
      } else {
        $asAND = $isShowing;
      }

      $result = true;

        foreach ($info['config'] as $key => &$value) {
          if (strpos($key, $keyPrefix.' ') === 0) {
            $name = substr($key, 5);
            $event = 'sidebars: show: '.$name;
            if (BEvent::Exists($event)) {
              $result &= BEvent::FireResult('bool', $event, array($value, $info));
              if ($asAND ? !$result : $result) { break; }
            }
          }
        }

      if ($isShowing ? !$result : $result) {
        return true;
      }
    }
  }

  static function GenerateFromCache(array &$info) {
    if ($cache = &$info['config']['cache'] and ($cache = BConfig::ToBool($cache))) {
      $cached = self::ReadCacheOf($info['id']);
      if ($cached and !BEvent::FireResult('bool', 'sidebars: is regenerated', array($info, $cached))) {
        self::MergeInfoInto($info, $cached);
        return true;
      } else {
        $cached and self::DeleteCacheOf($info['id']);
        $cached = false;
      }
    }
  }

    static function MergeInfoInto(array &$left, array &$right) {
      foreach ($left as $key => &$value) {
        if (isset($right[$key])) {
          if (is_array($right[$key])) {
            if (is_array($left[$key])){
              self::MergeInfoInto($left[$key], $right[$key]);
            } else {
              $value = $right[$key];
            }
          } else {
            $value = $right[$key];
          }
        }
      }

      $left += $right;
    }

    static function IsRegenerated(&$result, array $info, array $cached) {
      static $pf = 'regenerate ';

      if (!$result and $time = &$cached['cacheTime'] and filemtime($info['file']) > $time) {
        $result = true;
      }

      if (!$result and $ifEmpty = @$info['config']['regenerate if empty'] and
          BConfig::ToBool($ifEmpty)) {
        $content = &$cached['content'];
        if ( is_string($content) ? ("$content" === '') : empty($content) ) {
          $result = true;
          return;
        }
      }

      foreach ($info['config'] as $key => &$value) {
        if (strpos($key, $pf) === 0) {
          $name = substr($key, strlen($pf));
          $event = 'sidebars: regenerate: '.$name;

          if (BEvent::Exists($event)) {
            $res = BEvent::FireResult('bool', $event, array($info['config'], $cached));
          } else {
            $res = BEvent::FireResult('bool', 'sidebars: show: '.$name, array($value, $info));
          }

          if ($res) {
            $result = true;
            return;
          }
        }
      }
    }

    static function ApplyPostGenerationSettings(array &$info) {
      $icons = &$info['config']['icons'];
      if (isset($icons)) {
        is_array($icons) or $icons = explode(' ', " $icons");
        $icons = array_flip($icons);

        foreach ($info as $key => &$value) {
          if (substr($key, 0, 4) === 'icon') {
            $size = substr($key, 4);
            if (ltrim($size, '0..9') === '' and empty( $icons[$size] )) {
              $value = null;
            }
          }
        }
      }
    }

    static function StoreInCache(array &$info) {
      if ($info['config']['cache']) {
        self::WriteCacheOf($info['id'], $info);
        self::RefreshEventsOf($info);
      }
    }

      static function RefreshEventsOf(array $info) {
        $oldEvents = &$info['cacheOldEvents'];
        is_array($oldEvents) or $oldEvents = array();

        if ($new = &$info['config']['regenerate on events']) {
          $newEvents = explode(',', $new);
          foreach ($newEvents as &$one) { $one = trim($one); }
        } else {
          $newEvents = array();
        }

          foreach (array_diff($newEvents, $oldEvents) as $event) {
            BIndex::AddTo('SidebarEvents', $event, $info['id']);
          }

          foreach (array_diff($oldEvents, $newEvents) as $event) {
            BIndex::RemoveFrom('SidebarEvents', $event, $info['id']);
          }
      }

  static function ReadCacheOf($id) {
    return BIndex::SelectFrom('SidebarCache', $id);
  }

    static function WriteCacheOf($id, array $info) {
      $info['cacheTime'] = time();
      if (isset( $info['config']['regenerate on events'] )) {
        $info['cacheOldEvents'] = $info['config']['regenerate on events'];
      }

      BIndex::AddTo('SidebarCache', $id, $info);
    }

    static function DeleteCacheOf($id) {
      BIndex::RemoveFrom('SidebarCache', $id);
    }

  static function GenerateHtmlOf(array &$info) {
    if ($info['type'] === 'html') {
      $info['content'] = $info['rawContent'];
      return true;
    }
  }

    static function GenerateWikiOf(array &$info) {
      if ($info['type'] === 'wiki') {
        $content = WikiCache::Get($info['id'], $info['file'], $info['rawContent']);
        if ($content !== null) {
          $info['content'] = $content;
          return true;
        }
      }
    }

    static function GenerateByPlugin(array &$info) {
      if ($info['type'] === 'plugin') {
        $plugin = basename($info['file'], $info['ext']);
        $prefix = strtok($plugin, ' ');
        ltrim($prefix, '0..9') === '' and $plugin = strtok(null);
        $plugin = strtok($plugin, ' - ');

        $event = 'sidebars: generate: '.trim($plugin);
        BEvent::Fire($event, array(&$info));
      }
    }

  static function ShowOnPages($list, array &$info) {
    $cur = BPosts::CurrentEx();
    if ($cur) {
      is_array($list) or $list = explode(',', $list);

      foreach ($list as $page) {
        if (strtolower(trim($page)) === $cur) { return true; }
      }

      return false;
    }
  }

    static function ShowOnChance($chance, array &$info) {
      return $chance > mt_rand(0, 99);
    }

    static function ShowOnDates($list, array &$info) {
      is_array($list) or $list = explode(',', $list);

        $curTime = time();
        foreach ($list as $item) {
          $mask = strtok($item, '=');
          if ($mask) {
            $variants = explode(',', strtok(null));

            $cur = date(trim($mask), $curTime);
            foreach ($variants as &$var) {
              if (trim($var) === $cur) { return true; }
            }
          } else {
            @list($start, $end) = explode('-', $item, 2);
            isset($end) or $end = $item;

            $start = self::ParseDate($start, false);
            $end = self::ParseDate($end, true);

              if ($start === null or $end === null) {
                $info = "start: $start, end: $end, sidebar: ".var_export($info);
                throw new BException('One of sidebar \'show on dates\' range parts'.
                                     ' cannot be converted to timestamp.', $info);
              }

            if ($curTime >= $start and $curTime <= $end) { return true; }
          }
        }

      return false;
    }

      static function ParseDate($str, $maxSpan = false) {
        static $partMaxes = array('year' => null, 'month' => 12, 'day' => null,
                                  'hour' => 23, 'minute' => 59, 'second' => 59);

        $time = strtotime($str);
        if (is_int($time) and $time >= 0) {
          if ($maxSpan and preg_match(self::$datePartsRegExp, $str, $match)) {
            $parts = array_intersect_key($match, $partMaxes);

              foreach (array_diff_key($partMaxes, $parts) as $part => $max) {
                if (!isset($parts[$part])) {
                  if ($part === 'day') {
                    $firstMonthDay = mktime(0, 0, 0, $parts['month'] + 1, 1, $parts['year']);
                    $max = (int) date('d', $firstMonthDay);
                  }

                  $parts[$part] = $max;
                }
              }

            $new = mktime($parts['hour'], $parts['minute'], $parts['second'],
                          $parts['month'], $parts['day'], $parts['year']);
            $new > 0 and $time = $new;
          }

          return $time;
        }
      }

    static function ShowOnAgents($list, array &$info) {
      $cur = &$_SERVER['HTTP_USER_AGENT'];
      if (isset($cur)) {
        is_array($list) or $list = explode('|', $list);

        foreach ($list as $agent) {
          if (MatchWildcard($agent, $cur)) { return true; }
        }

        return false;
      }
    }

    static function ShowOnRobots($value, array &$info) {
      return BConfig::ToBool($value) === IsRobotAgent();
    }

    static function ShowOnIPs($list, array &$info) {
      $cur = &$_SERVER['REMOTE_ADDR'];
      if (isset($cur)) {
        is_array($list) or $list = explode(',', $list);

        foreach ($list as $ip) {
          if (MatchWildcardIP($ip, $cur)) { return true; }
        }

        return false;
      }
    }

    static function ShowOnPorts($list, array &$info) {
      $cur = &$_SERVER['HTTP_REMOTE_PORT'];
      if (isset($cur) and self::IsPort($cur)) {
        is_array($list) or $list = explode(' ', $list);

        foreach ($list as $port) {
          if (self::IsPort($port) and $port === $cur) { return true; }

          list($start, $end) = explode('-', $port, 2);
          if ((self::IsPort($start) or $start === '0') and self::IsPort($end) and
              ($cur >= $start) and ($cur <= $end)) {
            return true;
          }
        }

        return false;
      }
    }

      static function IsPort($port) { return is_numeric($port) and $port > 0; }

    static function ShowIfWereEvents($list, array &$info) {
      if (BEvent::$fireCounts) {
        is_array($list) or $list = explode(',', $list);

        foreach ($list as $event) {
          if (!empty( BEvent::$fireCounts[trim($event)] )) { return true; }
        }

        return false;
      }
    }

  static function RegenOnAge($age, array &$info) {
    if ($time = &$info['cacheTime']) {
      return $time + $age < time();
    }
  }

  // hooked by the loop at the end of events.php.
  static function OnEvent($event) {
    $sidebars = BIndex::SelectFrom('SidebarEvents', $event);
    if ($sidebars) {
      foreach ($sidebars as $sidebar) { self::DeleteCacheOf($sidebar); }
    } else {
      BIndex::RemoveFrom('SidebarEvents', $event);
    }
  }
}
