<?php
BEvent::Hook('on end', array('BIndex', 'Flush'));

BEvent::Hook('index: file load', array('BaseFileIndex', 'LoadFrom'));
BEvent::Hook('index: file save', array('BaseFileIndex', 'SaveTo'));

class BIndex {
  static $list = array();  // 'type' => array('index', ...), 'type' => ...
  static $type = 'file';   // lower case.

  static function Add($index, $type) {
    $type = strtolower($type);
    $index = strtolower($index);

    self::$list[$type][] = $index;

    BEvent::HookInstanceIfLoaded(self::ClassOf($index), 'Flush', "index: flush");
    BEvent::HookInstanceIfLoaded(self::ClassOf($index), 'Flush', "index: flush $type");
  }

  static function Flush($type = null) {
    $type and $type = ' '.strtolower($type);
    BEvent::Fire("index: flush$type");
  }

  static function Exists($index) {
    $index = strtolower($index);
    return array_search($index, self::$list[self::$type]) !== false;
  }

  static function ClassOf($index) {
    $index = ucfirst( strtolower($index) );
    return $index.ucfirst(self::$type).'Index';
  }

  static function Call($index, $method, $args, $argShift = 1) {
    $class = self::ClassOf($index);
    if (!class_exists($class)) { throw new BException("Index class $class is not defined."); }

    $args = array_slice($args, $argShift);
    return call_user_func_array(array(SingleInstanceOf($class), $method), $args);
  }

    static function SelectFrom($index, $arg_1 = null) { return self::Call($index, 'Select', func_get_args()); }
    static function AddTo($index, $arg_1 = null) { return self::Call($index, 'Add', func_get_args()); }
    static function RemoveFrom($index, $arg_1 = null) { return self::Call($index, 'Remove', func_get_args()); }
    static function RenameIn($index, $arg_1 = null) { return self::Call($index, 'Rename', func_get_args()); }
}

abstract class BaseFileIndex {
  static $loaded = array();
  // if true will flush index after each writing and reload before reading.
  static $noBuffering = array();  // true/false OR array('index' => true/false, ...)

  abstract function Flush();

  // can't use abstract func as PHP will complain about incompatible definitions.
  function Select() {
    throw new BException('Child classes must redefine BaseFileIndex->Select()');
  }
  function Add($key) {
    throw new BException('Child classes must redefine BaseFileIndex->Add()');
  }
  function Remove($key) {
    throw new BException('Child classes must redefine BaseFileIndex->Remove()');
  }
  function Rename($key, $newName) {
    throw new BException('Child classes must redefine BaseFileIndex->Rename()');
  }

  static function FileOf($index) {
    return BConfig::FileOf('index', BConfig::FromUTF8('file name', $index).'.php');
  }

  static function From($index, $key = null) {
    $index = self::Load($index);
    return $key === null ? $index : @$index[$key];
  }

  static function &Load($index) {
    if (!trim($index)) {
      throw new BException('Cannot load index - empty name given.');
    }

    $data = &self::$loaded[$index];
    empty( self::$noBuffering[$index] ) or $data = null;

    isset($data) or BEvent::Fire('index: file load', array($index, &$data));

    if (!is_array($data)) {
      if ($data === null) {
        $data = array();
      } else {
        throw new BException('Failed to load index '.$index);
      }
    }

    return $data;
  }

    static function LoadFrom($index, &$data) {
      $index = self::FileOf($index);
      if (is_file($index)) { $data = include($index); }
    }

  static function Save($index) {
    if (!trim($index)) {
      throw new BException('Cannot save index - empty name given.');
    }

    $data = &self::$loaded[$index];
    if (isset($data)) {
      if (is_array($data)) {
        BEvent::Fire('index: file save', array($index, &$data));
      } else {
        throw new BException("Failed to save index $index because it wasn't an array");
      }
    }
  }

    static function SaveTo($index, &$data) {
      $index = self::FileOf($index);
      MkDirOf($index);

      $export = "<?php\nreturn ".var_export($data, true).';';
      if (!is_int( file_put_contents($index, $export, LOCK_EX) )) {
        throw new BException('Cannot write index file '.$index);
      }
    }

  function Clear($index) { self::$loaded[$index] = array(); }
}

  abstract class BaseHashFileIndex extends BaseFileIndex {
    public $indexName;
    // false/null, keys, keys desc, values, values desc.
    public $sort = false;

    private $modified = false;

    function Flush() {
      if ($this->modified) {
        $this->modified = false;
        self::Save($this->indexName);
      }
    }

      function Modified() {
        $this->modified = true;
        if (!empty( self::$noBuffering[$this->indexName] ) ) {
          $this->Flush();
        }
      }

    function Select($key) { return self::From($this->indexName, $key); }

    function Add($key, $value) {
      $index = &self::Load($this->indexName);
      $needsSorting = !isset($index[$key]);
      $index[$key] = $value;

      $needsSorting and $this->Sort($index);
      $this->Modified();
    }

      function Sort(&$index) {
        $func = null;

        switch ($this->sort) {
        case 'keys':        $func = 'ksort'; break;
        case 'keys desc':   $func = 'krsort'; break;

        case 'values':      $func = 'asort'; break;
        case 'values desc': $func = 'arsort'; break;
        }

        if ($func) {
          if ($func[0] === 'a') {
            $first = array_values( array_slice($index, 0, 1) );
            if (!is_scalar($first[0]) and $first[0] !== null) {
              $func = null;
            }
          }

          $func and $func($index);
        }
      }

    function Remove($key) {
      $index = &self::Load($this->indexName);
      if (isset( $index[$key] )) {
        unset($index[$key]);
        $this->Modified();
      }
    }

    function Clear() {
      $index = &self::Load($this->indexName);
      if (!empty($index)) {
        $index = array();
        $this->Modified();
      }
    }

    function Rename($key, $newName) { $this->RenameKey($key, $newName); }

      function RenameKey($key, $newName) {
        $index = &self::Load($this->indexName);
        if (isset($index[$key])) {
          $index[$newName] = $index[$key];
          unset($index[$key]);

          $this->Sort($index);
          $this->Modified();
        }
      }
  }

    // a dual-side hash allowing access by index as well as by key. Both keys and values are unique.
    abstract class BaseArrayFileIndex extends BaseHashFileIndex {
      protected $byIndex;

      function Add($key) {
        $newIndex = $this->Select($key);
        if (!$newIndex) {
          $newIndex = $this->Count();
          while ($this->SelectByIndex(++$newIndex) !== null) { }

          parent::Add($key, $newIndex);
          $this->byIndex[$newIndex] = $key;
        }

        return $newIndex;
      }

        function Count() { return (int) count( self::Load($this->indexName) ); }

      function SelectByIndex($index) {
        if (!$this->byIndex) {
          $loaded = &self::Load($this->indexName);
          $this->byIndex = array_flip($loaded);
          if (count($this->byIndex) !== count($loaded)) {
            throw new BException("Dual-side hash index {$this->name} contains duplicate indexes.");
          }
        }

        return $index > 0 ? @$this->byIndex[$index] : null;
      }
    }

    abstract class BaseHashOfHashFileIndex extends BaseHashFileIndex {
      function Select($key = null, $subkey = null) {
        $result = self::From($this->indexName, $key);
        if (isset($key) and isset($subkey)) {
          return isset($result[$subkey]) ? $result[$subkey] : null;
        } else {
          return $result;
        }
      }

      function Add($key, $subkey, $value) {
        $index = &self::Load($this->indexName);
        $needsSorting = !isset($index[$key]);
        $index[$key][$subkey] = $value;

        $needsSorting and $this->Sort($index);
        $this->Modified();
      }

        function AddKey($key, $value) {
          $index = &self::Load($this->indexName);
          $needsSorting = !isset($index[$key]);
          $index[$key] = $value;

          $needsSorting and $this->Sort($index);
          $this->Modified();
        }

      function Remove($key, $subkey = null) {
        $index = &self::Load($this->indexName);
        if ($subkey === null) {
          if (isset($index[$key])) {
            unset($index[$key]);
            $this->Modified();
          }
        } elseif (isset( $index[$key][$subkey] )) {
          unset( $index[$key][$subkey] );
          $this->Modified();
        }
      }

      function Rename($key, $subkey, $newSubkey) {
        $index = &self::Load($this->indexName);
        if (isset($index[$key][$subkey])) {
          $index[$key][$newSubkey] = $index[$key][$subkey];
          unset($index[$key][$subkey]);

          $this->Sort($index);
          $this->Modified();
        }
      }
    }

    // represents a hash (key => value) where value is either a hash or an array.
    // example: array('a-tag' => array('my-post', 'another-post'), 'tag-2' => ...)
    // > This and all its descendants are made as one index file per one class only. <
    abstract class BaseHashOfArraysFileIndex extends BaseHashFileIndex {
      function Select($key = null) {
        $result = self::From($this->indexName, $key);
        return $result ? $result : array();
      }

      function Add($key, $value) {
        $index = &self::Load($this->indexName);
        if (!isset( $index[$key] ) or !in_array($value, $index[$key])) {
          $index[$key][] = $value;
          $this->Sort($index);
          $this->Modified();
        }
      }

      function Remove($key, $value = null) {
        $index = &self::Load($this->indexName);
        if ($value === null) {
          unset($index[$key]);
          $this->Modified();
        } elseif (isset( $index[$key] )) {
          $i = array_search($value, $index[$key]);
          if ($i !== false) {
            unset( $index[$key][$i] );
            if (empty( $index[$key] )) { unset($index[$key]); }
            $this->Modified();
          }
        }
      }

      // function ($key, $newKey)  or  function ($key, $value, $newValue)
      function Rename($key, $value, $newValue = null) {
        if ($newValue === null) {
          $this->RenameKey($key, $value);
        } else {
          $index = &self::Load($this->indexName);
          if (isset( $index[$key] )) {
            $i = array_search($value, $index[$key]);
            if ($i !== false) {
              $index[$key][$i] = $newValue;
              $this->Modified();
            }
          }
        }
      }
    }

    // Represents e.g. a number of referring links with number of visitors. Is kept sorted.
    // array('a-post' => array('http://referral/1' => 102, '2nd' => 30))
    abstract class BaseHashOfIntHashesFileIndex extends BaseHashFileIndex {
      public $allowNegative = false;
      public $reverseSort = false;

      function Add($key, $subkey) {     // => index[key][subkey] += 1
        $index = &self::Load($this->indexName);
        if (!isset( $index[$key][$subkey] )) {
          $index[$key][$subkey] = 0;
        }

        ++$index[$key][$subkey];
        $this->Sort($index, $key);
        $this->Modified();
      }

        function Sort(&$index, $key) {
          $func = $this->reverseSort ? 'arsort' : 'asort';
          $func( $index[$key] );
        }

      function Remove($key, $subkey = null) {
        $index = &self::Load($this->indexName);
        if ($subkey === null) {
          unset($index[$key]);
          $this->Modified();
        } else {
          if (isset( $index[$key][$subkey] )) {
            $value = $index[$key][$subkey] - 1;
          } else {
            $value = -1;
          }

          $this->allowNegative or $value = max(0, $value);

          // no need to write 0 if it doesn't exist since it's already 0 by default:
          if ($value or isset( $index[$key][$subkey] )) {
            if ($value) {
              $index[$key][$subkey] = $value;
            } else {
              unset( $index[$key][$subkey] );
            }

            $this->Modified();
          }
        }
      }

      function Rename($key, $subkey, $newSubkey) {
        $index = &self::Load($this->indexName);
        if (isset($index[$key][$subkey])) {
          $index[$key][$newSubkey] = $index[$key][$subkey];
          unset($index[$key][$subkey]);
          $this->Sort($index, $key);
          $this->Modified();
        }
      }
    }

      // a hash with timestamp values; used, for example, in post/comment date indexes.
      abstract class BaseHashOfDatesFileIndex extends BaseIntHashFileIndex {
        public $sort = 'values';

        static function GetAt($path, &$array) {
          $path = (array) $path;
          $last = array_pop($path);

          foreach ($path as $part) {
            if (is_array($array)) {
              $array = &$array[$part];
              isset($array) or $array = array();
            } else {
              return;
            }
          }

          return array(&$array, $last);
        }

        function Select($key = null, $offset = null) {
          $index = self::GetAt($key, self::Load($this->indexName));

          if ($index !== null) {
            if (is_numeric($offset)) {
              $keys = array_keys($index[0]);
              $i = array_search($index[1], $keys);
              return is_int($i) ? @$keys[$i + $offset] : null;
            } else {
              $index = $index[1] === null ? $index[0] : @$index[0][$index[1]];

              if ($offset === null or $index === null) {
                return $index;
              } else {
                $keys = array_keys($index);

                switch ($offset) {
                case 'first':   return reset($keys);
                case 'last':    return end($keys);
                case 'random':  return $keys[ array_rand($keys) ];
                default:
                  throw new BException(get_class($this).'->Select() $offset.', $Offset);
                }
              }
            }
          }
        }

        function Add($key, $timestamp = null) {
          $index = self::GetAt($key, self::Load($this->indexName));

          if ($index !== null) {
            $index[0][$index[1]] = $timestamp ? $timestamp : time();
            $this->Sort($index[0], $index[1]);
            $this->Modified();
          }
        }

        function Remove($key) {
          $index = self::GetAt($key, self::Load($this->indexName));

          if ($index !== null) {
            unset( $index[0][$index[1]] );
            $this->Modified();
          }
        }
      }

    // example (views per post index): array('my-post' => 10, 'another-post' => 34)
    abstract class BaseIntHashFileIndex extends BaseHashFileIndex {
      public $allowNegative = false;

      function Add($key, $amount = 1) {
        $index = &self::Load($this->indexName);
        $hadKey = isset( $index[$key] );

        $hadKey or $index[$key] = 0;
        $index[$key] += $amount;

        $hadKey or $this->Sort($index);
        $this->Modified();
      }

      function Remove($key, $amount = 1) {
        $index = &self::Load($this->indexName);
        $value = isset( $index[$key] ) ? $index[$key] - $amount : (-1 * $amount);
        $this->allowNegative or $value = max(0, $value);

        if ($value or isset( $index[$key] )) {
          if ($value) {
            $index[$key] = $value;
          } else {
            unset( $index[$key] );
          }

          $this->Modified();
        }
      }
    }

      // simply a hash of true/false values (false's are removed): array('my post.wiki' => true).
      abstract class BaseBoolHashFileIndex extends BaseIntHashFileIndex {
        function Select($checkForKey = null) {
          $result = parent::Select($checkForKey);
          return $checkForKey === null ? array_keys($result) : (bool) $result;
        }

        function Add($key) {
          $this->Select($key) or parent::Add($key);
        }

        function Remove($key = null) {
          if ($key === null) {
            Clear($this->indexName);
          } else {
            $index = &self::Load($this->indexName);
            if (isset( $index[$key] )) {
              unset( $index[$key] );
              $this->Modified();
            }
          }
        }
      }
