<?php
include_once 'base_formatters.php';

abstract class UWikiBaseElement {
  public $className, $elementName, $elementID;
  public $doc, $settings, $strings, $pager, $style;  // are initialized by UWikiDocument->NewElement().
  public $treePosition = array();

  // $kind: heading, footnote, anchor, format, list, listItem, paragraph, line, link.
  // Block or inline formats/footnotes are determined using $isBlock.
  public $kind;
  public $isBlock = false;
  public $isEditable = true;  // whether this element can be edited (e.g. using Inlinedit) or not.
  public $runsImmediately = false;
  public $isFormatter = false;  // an info value meaning that markup accepts user-input text ($raw).
  public $isAction = true;  // similar to above but means that markup allows {{Markup ...}} form.
  public $parameterOrder = array();
  public $parameterAliases = array();   // in addition to those from $settings->strings.
  public $anchorClass = 'UWikiAnchorElement';

  // $source includes start/end tokens (**bold**), $originalRaw - element's content only (bold).
  public $source, $raw, $originalRaw;
  public $children = array();

  public $anchor;   // assigned anchor element, if any (see Anchorize()).
  public $htmlTag, $html5Tag;
  public $isSingleLineHTML = false;
  public $isSingleHtmlTag = false;
  public $htmlClasses = array();
  public $htmlAttributes = array();   // keys returned by HtmlAttributes() override keys here.

  public $serializeList = array('treePosition');
  public $serializeHash = array('htmlClasses', 'htmlAttributes');
  public $serializeBool = array('isBlock', 'isSingleLineHTML', 'isSingleHtmlTag');
  public $serializeInt = array('elementID');
  public $serializeStr = array('htmlTag', 'html5Tag');
  public $serializeElement = array('anchor');
  public $serializeDocument = array();

  // Are called by UWikiDocument or other host (Include, etc.) on its root element (only root).
  function BeforeParsing() { $this->doc->BeforeParsing(); }
  function AfterParsing() { $this->doc->AfterParsing(); }
  function BeforeRenderingInto($format) { $this->doc->BeforeRenderingInto($format); }
  function AfterRenderingInto($format, &$result) { $this->doc->AfterRenderingInto($format, $result); }

  function IsDynamic($format, $params) { return false; }
  // 0 - lowest priority, UWikiMaxPriority - highest.
  function Priority($format, $params) { return (int) UWikiMaxPriority * 0.5; }
  // gets called if doesn't $runsImmediately by UWikiFormat after this element was put in chain.
  function WasFound($format, $params) { }

  function HasBlockChildren() { return $this->isBlock; }
  function Attachments() { return array(); }

  function IsRoot() {
    return substr($this->elementName, strpos($this->elementName, '_') + 1) === 'Root';
  }

    function MarkupName() { return UWikiDocument::MarkupNameFrom($this->elementName); }
    function ElementName() { return UWikiDocument::ElementNameFrom($this->elementName); }

  function PrepareParams($params) {
    $params = $this->ApplyParameterAliasesTo($params);
    $params = $this->ApplyParameterOrderTo($params);
    return $params;
  }

    function ApplyParameterAliasesTo($params) {
      $result = array();

      foreach ($params as $name => &$value) {
        $real = $this->RealParameterNameBy($name);
        self::IsEmptyStr($real) and $real = $name;
        $result[$real] = &$value;
      }

      return $result;
    }

      function RealParameterNameBy($alias) {
        $alias = mb_strtolower($alias);

        $real = &$this->parameterAliases[$alias];
        self::IsEmptyStr($real) and $real = &$this->strings[$this->elementName.': param '.$alias];

        return $real;
      }

    // %%(fmt paramByOrder) => %%(fmt nameOfParamWithOrder0=paramByOrder)
    function ApplyParameterOrderTo($params) {
      $names = array_keys($params);
      foreach (array_values($params) as $i => $value) {
        if ($value === true and isset( $this->parameterOrder[$i] )) {
          $params[ $this->parameterOrder[$i] ] = $names[$i];
          unset( $params[$names[$i]] );
        } else {
          break;
        }
      }

      return $params;
    }

  function __construct() {
    $this->className = get_class($this);
    $this->elementName = substr($this->className, 1);

    $allProps = array('list' => &$this->serializeList,  'hash' => &$this->serializeHash,
                      'bool' => &$this->serializeBool,  'int' => &$this->serializeInt,
                      'str' => &$this->serializeStr,    'element' => &$this->serializeElement,
                      'document' => &$this->serializeDocument);
    $this->SetupSerialize($allProps);
  }

    function SetupSerialize(array &$props) { }

  function NewElement($class, $atPosition = 'child') {
    $obj = $this->doc->NewElement($class);

      $thisPos = $this->treePosition;
      $thisPos[] = ($atPosition === 'child' ? count($this->children) : array_pop($thisPos)) + 1;
      $obj->treePosition = $thisPos;

    switch ($obj->kind) {
    case 'heading':     $prop = 'headings'; break;
    case 'footnote':    $obj->isBlock and $prop = 'blockFootnotes'; break;
    }

      if (isset($prop)) {
        $id = $obj->TreePosID();
        $this->settings->all[$prop][$id] = $obj;
        $prop = &$this->settings->$prop;
        $prop[$id] = $obj;
      }

    return $obj;
  }

  function TreePosID() {
    $pos = $this->treePosition;
    // ksort() sorts based on lexical order thus "10" goes before "2". Pad length
    // of 4 allows 9999 concurrent elements - must be enough.
    foreach ($pos as &$one) { $one = str_pad($one, 4, '0', STR_PAD_LEFT); }
    return join('.', $pos);
  }

  function SetRaw($raw) { $this->raw = $this->originalRaw = $raw; }
  function SetRawAndSource($raw) { $this->source = $this->raw = $this->originalRaw = $raw; }

  function Parse() { $this->SetSettingsFrom($this->raw); }

    function SetSettingsFrom(&$raw) { }

  function Anchorize($baseAnchorName = null) {
    if (self::IsEmptyStr($baseAnchorName)) {
      $name = $this->settings->anchors->GenerateFor($this);
    } else {
      $name = $this->settings->anchors->UniqueNameFrom($baseAnchorName);
    }

    if (!self::IsEmptyStr($name) and $class = $this->anchorClass) {
      if ($this->anchor) {
        $anchor = $this->anchor;
      } else {
        $anchor = $this->anchor = $this->NewElement($class);
      }

      $anchor->PointTo($this);
      $anchor->SetNameAndRegister($name);
    }
  }

    function AnchorizeIfNeeds($baseAnchorName = null) {
      !$this->anchor and $this->Anchorize($baseAnchorName);
    }

  function AllToHTML() {
    return $this->SelfToHtmlWith( $this->isSingleHtmlTag ? '' : $this->ChildrenToHTML() );
  }

    function ChildrenToHTML($withAnchor = true) {
      $html = '';
      $renderedDocs = array();

        $isBlock = $this->isBlock;
        // children might get updated along the way so we use for + count() instead of foreach.
        for ($i = 0; isset($this->children[$i]); ++$i) {
          $child = $this->children[$i];
          if ($child->doc !== $this->doc) {
            $renderedDocs[] = $child->doc;
            $child->doc->BeginRenderingInto('html');
          }

          if (!$isBlock and $child->isBlock) {
            $html .= $child->InlineHTML('null message');
          } else {
            $html .= $child->AllToHTML();
          }
        }

      foreach ($renderedDocs as $doc) { $doc->EndRenderingInto('html', $html); }

      if ($this->anchor and $withAnchor) { $html .= $this->anchor->AllToHTML(); }
      return $html;
    }

      // $ifBlocks: message, exception, null (return null); also 'null message'
      //            and 'null exception' that will work even if $this had no
      //            child nodes (by default null is returned).
      function InlineHTML($ifBlocks = 'message') {
        $inline = $this->FindInline();
        if ($inline) {
          return $inline->ChildrenToHTML();
        } else {
          $selfEmpty = $inline === null;
          if (strtok($ifBlocks, ' ') === 'null') {
            $ifBlocks = strtok(null);
          } elseif ($selfEmpty) {
            return null;
          }

          switch ($ifBlocks) {
          case 'message':   return $this->NewElement('UWikiInliningBlocksError')->AllToHTML();
          case 'exception': throw new EUWikiInliningBlocks($this);
          }
        }
      }

    function SelfToHtmlWith($contents) {
      $this->doc->IsBeingParsed() or $this->doc->EnsureIsRenderedInto('html');
      return $this->SelfToHtmlUnchecking($contents);
    }

      protected function SelfToHtmlUnchecking($contents) {
        $attrs = ((array) $this->SelfHtmlAttributes()) + $this->htmlAttributes;

        $classes = array_merge($this->settings->globalHtmlClasses, $this->htmlClasses);
        if ($classes) {
          $attrs['class'] = isset( $attrs['class'] ) ? $attrs['class'].' ' : '';
          $attrs['class'] .= join(' ', array_unique($classes));
        }

        $htmlPrefix = $htmlSuffix = '';

        if ($handlers = &$this->doc->onHtmlTagHooks[$this->elementName]) {
          UWikiDocument::CallAll($handlers, array($this, &$attrs, &$contents, &$htmlPrefix, &$htmlSuffix));
        }
        if ($handlers = &$this->doc->onHtmlTagHooks['any']) {
          UWikiDocument::CallAll($handlers, array($this, &$attrs, &$contents, &$htmlPrefix, &$htmlSuffix));
        }

        $attrStr = '';
          foreach ($attrs as $name => $value) {
            if ($name[0] === '!') {
              $name = substr($name, 1);
            } else {
              $value = self::QuoteHTML($value, ENT_COMPAT);
            }
            self::IsEmptyStr($value) or $attrStr .= " $name=\"$value\"";
          }

        $nlIfBlock = ((!$this->isSingleLineHTML and $this->isBlock) ? "\n" : '');

        $tag = $this->settings->enableHTML5 ? $this->html5Tag : null;
        $tag or $tag = $this->htmlTag;
        if ($this->isSingleHtmlTag) {
          $html = $tag ? "<$tag$attrStr />" : '';
        } else {
          $html = $tag ? "<$tag$attrStr>$nlIfBlock$contents$nlIfBlock</$tag>" : $contents;
        }

        if ($attachments = $this->Attachments()) {
          $this->doc->attachments = array_merge_recursive($this->doc->attachments, $attachments);
        }

        return $htmlPrefix.$html.$htmlSuffix.$nlIfBlock.$nlIfBlock;
      }

        function SelfHtmlAttributes() { return array(); }

  function SerializeTo(UWikiSerializer $ser) {
    $buf = '';

    $serList = $this->serializeList;

      // note: watch for var name collision when using &by-references.
      foreach ($serList as &$listProp) { $listProp = $this->$listProp; }

      foreach ($this->serializeHash as $prop) {
        $serList[] = array_keys($this->$prop);
        $serList[] = array_values($this->$prop);
      }

    if ($serList) {
      // Packed lists:  [W]  item count of list 1
      //                [W]  ...
      //                [W]  item count of the last list
      //                [DW] length of item 1 of list 1
      //                [DW] length of next item (or the next list's first item)
      //                [DW] ...
      //                [S]  bytes of item 1 of list 1
      //                [S]  ...
      // Hashes are stored as if they were 2 lists (keys and values).
      $toPack = array('v'.count($serList).'V*');

      foreach ($serList as &$list) { $toPack[] = count($list); }

      foreach ($serList as &$list) {
        $buf .= join($list);
        foreach ($list as &$item) { $toPack[] = strlen($item); }
      }

      $buf = call_user_func_array('pack', $toPack).$buf;
      $count = fwrite($ser->handle, $buf, strlen($buf));
      isset($buf[$count + 1]) and $ser->WriteError($count, strlen($buf));
    }

    $buf = '';

      foreach ($this->serializeBool as $prop) {
        $buf .= chr( $this->$prop ? UWikiSerializer::BoolTrue : UWikiSerializer::BoolFalse );
      }

      $toPack = array('V*');
      $tail = '';

        foreach ($this->serializeInt as $prop) { $toPack[] = $this->$prop; }

        foreach ($this->serializeStr as $prop) {
          $toPack[] = strlen($this->$prop);
          $tail .= $this->$prop;
        }

    $buf .= call_user_func_array('pack', $toPack).$tail;
    $count = fwrite($ser->handle, $buf, strlen($buf));
    isset($buf[$count + 1]) and $ser->WriteError($count, strlen($buf));

    foreach ($this->serializeElement as $prop) { $ser->WriteElement($this->$prop); }
    foreach ($this->serializeDocument as $prop) { $ser->WriteDocument($this->$prop); }
  }

  function UnserializeFrom(UWikiUnserializer $ser) {
    $serList = array();

      foreach ($this->serializeList as $prop) {
        $this->$prop = array();
        $serList[] = &$this->$prop;
      }

      foreach ($this->serializeHash as $prop) {
        $serList[] = array();
        $serList[] = array();
      }

    if ($listCount = count($serList)) {
      if ($listCount > UWikiUnserializer::MaxArrayCount) {
        $ser->TooLarge($listCount, UWikiUnserializer::MaxArrayCount);
      } else {
        // note: unpack() returns 1-based index keys.
        $counts = unpack('v*', fread($ser->handle, $listCount * 2));
        if (!isset( $counts[$listCount] )) {
          $ser->PrematureEOF($listCount * 2, count($counts) * 2);
        }

        if ($sum = array_sum($counts)) {
          $lengths = unpack('V*', fread($ser->handle, $sum * 4));
          if (!isset( $lengths[$sum] )) {
            $ser->PrematureEOF($sum * 4, count($lengths) * 4);
          }

          foreach ($serList as $propI => &$list) {
            for (; --$counts[$propI + 1] >= 0; ) {
              $length = array_shift($lengths);
              if ($length > UWikiUnserializer::MaxStringLength) {
                $ser->TooLarge($length, UWikiUnserializer::MaxStringLength);
              }

              $str = fread($ser->handle, $length);
              if ($str === false or !isset($str[$length - 1])) {
                $ser->PrematureEOF($length, strlen($str));
              }

              $list[] = $str;
            }
          }
        }

        $hashes = array_slice($serList, count($this->serializeList));
        foreach ($this->serializeHash as $propI => $prop) {
          if (empty( $hashes[$propI * 2] )) {
            $this->$prop = array();
          } else {
            $this->$prop = array_combine($hashes[$propI * 2], $hashes[($propI * 2) + 1]);
          }
        }
      }
    }

    if ($count = count($this->serializeBool)) {
      if ($count > UWikiUnserializer::MaxArrayCount) {
        $ser->TooLarge($count, UWikiUnserializer::MaxArrayCount);
      }

      $unpacked = unpack('C*', fread($ser->handle, $count));
      if (!isset( $unpacked[$count] )) {
        $ser->PrematureEOF($count, count($unpacked));
      } else {
        foreach ($this->serializeBool as $propI => $prop) {
          $value = $unpacked[$propI + 1] === UWikiSerializer::BoolTrue;
          if (!$value and $unpacked[$propI + 1] !== UWikiSerializer::BoolFalse) {
            $ser->Warning('bool prop value', "wrong bool value for prop '$prop' - got $byte but expected either ".
                                             self::BoolTrue.' (for true) or '.self::BoolFalse.' (for false)');
            $value = $value !== 0;
          }

          $this->$prop = $value;
        }
      }
    }

    $count = count($this->serializeInt) + count($this->serializeStr);
    if ($count) {
      if ($count > UWikiUnserializer::MaxArrayCount) {
        $ser->TooLarge($count, UWikiUnserializer::MaxArrayCount);
      }

      $unpacked = unpack('V*', fread($ser->handle, $count * 4));
      if (!isset( $unpacked[$count] )) {
        $ser->PrematureEOF($count * 4, count($unpacked) * 4);
      } else {
        foreach ($this->serializeInt as $propI => $prop) {
          $this->$prop = $unpacked[$propI + 1];
        }

        $lengths = array_slice($unpacked, count($this->serializeInt));
        foreach ($this->serializeStr as $propI => $prop) {
          $length = $lengths[$propI];

          if ($length) {
            if ($length > UWikiUnserializer::MaxStringLength) {
              $ser->TooLarge($lengths[$propI], UWikiUnserializer::MaxStringLength);
            }

            $str = fread($ser->handle, $length);
            if (!isset( $str[$length - 1] )) {
              $ser->PrematureEOF($length, strlen($str));
            }

            $this->$prop = $str;
          } else {
            $this->$prop = '';
          }
        }
      }
    }

    foreach ($this->serializeElement as $prop) {
      $this->$prop = $ser->ReadElement();
    }

    foreach ($this->serializeDocument as $prop) {
      $this->$prop = $ser->ReadDocument();
    }
  }


  /* Utility methods: */

  static function CountLeftWhiteSpace($text, $startPos = 0) {
    $count = $startPos;
    while (self::IsWhiteSpace(@$text[$count++])) { }
    return $count - 1 - $startPos;
  }

    // Whitespace = all chars \01..' ', \0 isn't included so null vars won't match.
    static function IsWhiteSpace($char) {
        $code = ord($char);
      return $code > 0 and $code <= 32;
    }

  static function CountLeftCharIn($text, $charToCount, $startPos = 0) {
    $count = $startPos;
    while (@$text[$count++] === $charToCount) { }
    return $count - 1 - $startPos;
  }

  // Sets default style if $userStyle == false, resolves aliases and returns final style name to be used.
  function RealStyleBy($userStyle, $ifUnset = 'default') {
    self::IsEmptyStr($userStyle) and $userStyle = &$this->settings->styleAliases[$this->elementName];

    $aliasedTo = &$this->settings->styleAliases[ mb_strtolower($userStyle) ];
    self::IsEmptyStr($aliasedTo) or $userStyle = $aliasedTo;

    return mb_strtolower( self::IsEmptyStr($userStyle) ? $ifUnset : $userStyle );
  }

    function DefaultStyle($ifUnset = null) {
      if ($ifUnset === null) {
        if ($this->isAction) {
          $ifUnset = 'notexistingformat';
        } elseif ($this->isFormatter) {
          $ifUnset = 'pre';
        } else {
          $ifUnset = 'default';
        }
      }

      return $this->RealStyleBy(null, $ifUnset);
    }

    function IsStyleDisabled($style) {
      $real = $this->RealStyleBy($style);
      $isAliasDisabled = $real === '-';
      return $isAliasDisabled or $this->RealStyleBy($real) === '-';
    }

  static function &DeleteSubstrIn(&$str, $pos, $length = 1) {
    $pos >= 0 or $pos = strlen($str) + $pos;
    $str = substr($str, 0, $pos) . substr($str, $pos + $length);
    return $str;
  }

  static function DeleteIfUnevenTail($char, &$text) {
    $escCount = strlen($text) - strlen(rtrim($text, $char));
    if ($escCount % 2 === 1) {
      $escCount > 0 and $text = substr($text, 0, -1);
      return true;
    }
  }

  static function &LastIn(&$array) {
    $last = array_slice($array, -1);  // don't use count() because $array may have non-numeric keys.
    return $last[0];
  }

  static function IsEmptyStr(&$value) { return "$value" === ''; }

  static function PathType($path) {
    if (strpos(substr($path, 0, 15), '://') !== false) {
      return 'ext';
    } elseif (strpbrk(substr($path, 0, 1), '\\/') !== false or
              (substr($path, 1, 1) === ':' and ltrim($path[0], 'a..zA..Z') === '')) {
      return 'abs';
    } else {
      return 'rel';
    }
  }

    static function IsPath($ofType, $path) {
      return self::PathType($path) === $ofType;
    }

    function ToAbsolutePath($path) {
      switch (self::PathType($path)) {
      case 'rel': return $this->settings->BaseUrlWithRoot().$path;
      case 'abs': return $this->settings->rootURL.ltrim($path, '\\/');
      default:    return $path;
      }
    }

  function UnindentBlockIfNeeds($raw) {
    return $this->settings->unindentBlocks ? self::UnindentBlock($raw) : $raw;
  }

    static function UnindentBlock($raw) {
      $raw = trim($raw, "\r\n");
      $nlCount = preg_match_all('/(\r?\n)+/u', $raw, $matches) + 1;
      $unindented = str_replace("\n  ", "\n", "\n".$raw, $replCount);
      return $nlCount === $replCount ? substr($unindented, 1) : $raw;
    }

  static function QuoteHTML($str, $quoteStyle = ENT_NOQUOTES, $doubleEncode = true) {
    return htmlspecialchars($str, $quoteStyle, 'UTF-8', $doubleEncode);
  }

  protected static $SimpleFmtArgs;
  // taken from StdTpl's templater.php. Also exists in blog's utils.php.
  static function SimpleFmt($pattern, $params, $char = '$') {
    $params = (array) $params;

    if ($pattern === $char) {
      return $params[0];
    } elseif (strpos($pattern, $char) === false) {
      return $pattern;
    } else {
      self::$SimpleFmtArgs = $params;
      $char = preg_quote($char, '/');
      return preg_replace_callback("/$char($char?)/u", array(__CLASS__, 'SimpleFmtCallback'), $pattern);
    }
  }

    static function SimpleFmtCallback($match) {
      if ($match[1] === '') {
        return count(self::$SimpleFmtArgs) > 1 ? array_shift(self::$SimpleFmtArgs) : self::$SimpleFmtArgs[0];
      } else {
        return $match[1];
      }
    }

  function BuildListFrom($items, $padItemClass, $groupClass, $firstIndex = 1) {
    $listRoot = $this->NewElement($groupClass);
    $listRoot->level = $firstIndex;
    $listRoot->Group($items[0]);

    $stack = array($listRoot);  // The deepest item has index 0.

    $prevLevel = $firstIndex - 1;
    for ($i = 0; isset($items[$i]); ++$i) {
      $item = $items[$i];

        while ($item->level > $prevLevel + 1) {
          $padItem = $this->NewElement($padItemClass);
          $padItem->level = ++$prevLevel;
          $padItem->Pad($item);

          array_splice($items, $i++, 0, array($padItem));
        }

      $prevLevel = $item->level;
    }

    while ($item = array_shift($items)) {
      $level = $item->level;

        while ($level > $stack[0]->level) {
          $parent = self::LastIn($stack[0]->children);

            $block = $parent->NewElement($groupClass);
            $block->level = count($stack) + $firstIndex;
            $block->Group($item);

          $parent->children[] = $block;
          array_unshift($stack, $block);
        }

        while ($level < $stack[0]->level) { array_shift($stack); }

      $doInsert = $this->PrepareItem($item, $stack);
      if ($doInsert === null or $doInsert) { $stack[0]->children[] = $item; }
    }

    return $listRoot;
  }

  // returns null if there were no children, false - if no inline elements were found
  // or the object of inline element otherwise.
  function FindInline() {
    $parent = $this;

    if ($parent->children) {
      $firstInline = $parent->children[0];
      while ($firstInline and $firstInline->isBlock) {
        $parent = $firstInline;
        $firstInline = &$firstInline->children[0];
      }

      return $firstInline ? $parent : false;
    }
  }

  function Deprecated($feature, $desc = null) {
    $this->doc->Deprecated($feature, $this->className, $desc);
  }

  // must be called after parse and before render phase - e.g. after unserializing a document.
  function ReplaceChildByRunning(UWikiFormat $format, UWikiBaseElement $child) {
    $i = array_search($child, $this->children, true);
    if ($i === false) {
      throw new EUverseWiki('Child element to replace not found in parent element');
    } else {
      unset( $this->children[$i] );
    }

    $format->ReattachTo($this);
    $format->RunImmediateFormats();
    $childCount = count($this->children);
    $format->ApplyAll();

    return array_slice($this->children, $childCount);
  }

  function FormatterError($markup) {
    $error = $this->NewElement('UWikiFormatterError');
    $error->format = array(ucfirst($markup));
    return $error;
  }
}

  abstract class UWikiBaseAction extends UWikiBaseElement {
    function Parse() {
      if ($format = &$this->settings->format) {
        $format->appendMe = false;
        $this->isBlock = $format->blockExpected;
        return $this->Execute($format, $format->current['params']);
      }
    }

      // $params is simply a shurtcut to $format->params.
      abstract function Execute($format, $params);
  }

  class UWikiAnchorElement extends UWikiBaseElement {
    public $kind = 'anchor';
    public $htmlTag = 'a';
    public $htmlClasses = array('anchor');

    // do not access it directly even from within the descendants. It's only made
    // protected to let Un/serialization methods access it.
    protected $name;
    protected $linked;

    function SetupSerialize(array &$props) {
      parent::SetupSerialize($props);

      $props['str'][] = 'name';
      $props['element'][] = 'linked';
    }

    function Name() { return $this->name; }

    function PointTo($element) {
      $this->linked = $element;

      if (!self::IsEmptyStr($this->name)) {
        $element or $element = $this;
        $this->settings->anchors->ReRegister($this->name, $element);
      }
    }

    function SetNameAndRegister($name) {
      $this->name = $name;
      $this->settings->anchors->Register($name, $this->linked ? $this->linked : $this);
    }

      // "exact" means that the name shouldn't be prepared using normal rules
      // (replacing wrong chars into '_' and so on).
      function SetExactAnchorName($name) {
        $name = mb_strtolower(trim($name));

        if ($this->settings->anchors->IsTaken($name)) {
          $index = 1;
          while ($this->settings->anchors->IsTaken($name.'_'.++$index)) { }
          $name .= '_'.$index;
        }

        $this->SetNameAndRegister($name);
      }

      function SetUniqueNameFrom($name) {
        $this->SetNameAndRegister( $this->settings->anchors->UniqueNameFrom($name, $this) );
      }

    function SelfToHtmlWith($contents) {
      $this->isBlock and $this->htmlClasses[] = 'block-anchor';
      return parent::SelfToHtmlWith('');
    }

      function SelfHtmlAttributes() {
        $name = $this->settings->anchorPrefix.mb_strtolower($this->Name());
        $href = $this->settings->pageForSelfAnchorLinks."#$name";
        return array('name' => $name, 'href' => $href, 'title' => "#$name");
      }
  }
