<?php
/*
  == {{Include}} ==
  This action merging a page into current document. It also corrects headers so that a 1-st level
  header in the embedded page becomes 3-rd level if {{Include}} itself was located under 2-nd level heading.

  Extension is optional; if specified it must equal to format or it'll be considered part of file name.
  If this is the case or no extension supplied format's extension will be appended. 'txt' and 'wiki'
  extensions (in this priority order) are synonyms of 'wacko'.

  Usage:  {{Include See also}}
          {{Include page=Embed_this.wacko}}
          {{Include BB samples, format=bbcode}}
          {{Include Localization, tree=top}}     - inserts doc at the same level as {{Include}} (not
                                                   below it) so headers are corrected by 1 less than
                                                   they normally would.
          {{Include Appendix A.wiki, isolate=1}} - completely abstract the included page from owner doc.
                                                   No header correction is performed either.

  You can also use Include as a formatter (makes sense when isolate=1 since without it
  it's usually the same as just using %%(format) instead of %%(Include format=format)).

    %%(Include isolate=1, format=wacko)
      == Heading of first level ==
    %%

  == {{RunDoc}} ==
  This is a version of {{Include}} that doesn't include the document into the
  current doc - it only formats it. {{RunDoc}} executes immediately as it was
  encountered in the document source which makes it appropriate for "running"
  docs containing control actions or something else that doesn't affect the
  display of source doc directly.

  Because {{RunDoc}} is ran immediately format chain doesn't apply to it, thus
  you can't do {{RunDoc; xml}} but only {{RunDoc format=xml}} (as with {{Include}}).

    For example, let's say you want to define a list of terms that are used in
    several of your documents. To do so you could repeat them in all docs like
    (?Term Description?) or {{Term Term=Description}}

    However, if you use (?Term?)s and {{Include}} you'll see that terms in parent
    doc were not replaced - that's because (?Term?) works for current (embedded)
    doc. If you use {{Term}} even with {{TopDoc}} it won't work either because of
    the way formatters are invoked - it won't affect the parent doc too.

      As a workaround you could use 2 embedded docs: one for terms and one for
      actual document contents - since both are [{Included}} terms are processed
      first and then the document itself. But this usually isn't very convenient.

    To solve this you can use {{RunDoc}} instead of {{Include}}:
    ==Main doc==
      {{RunDoc terms}}
      HTML, RTFM.
    ==Terms==
      {{TopDoc; Term HTML=HyperText Markup Language, RTFM=...}}
*/

UWikiDocument::$loadedHandlers['wacko']['linkOnRender'][] = array('Uinclude_Root', 'RelinkIfEmbedded');

class Uinclude_Root extends UWikiBaseAction {
  public $isFormatter = true;

  public $parameterOrder = array('page', 'format');
  public $parameterAliases = array('страница' => 'page', 'формат' => 'format', 'дерево' => 'tree',
                                   'изолировать' => 'isolate');

  public $page;

  static function RelinkIfEmbedded($linkObj) {
    if (!$linkObj->IsExternal() and !$linkObj->IsAnchorLink()) {
      $linkToAnchor = $linkObj->LocalAnchor();

        $format = $linkObj->settings->format;
        $topmost = $format ? $format->topmostDoc : $linkObj;

      if ($inc = &$topmost->settings->Included[$linkObj->LocalPath()]) {
        if (self::IsEmptyStr($linkToAnchor) and $inc['anchor']) {
          $linkToAnchor = $inc['anchor']->Name();
        }

        $url = Uwacko_AnchorChar.$topmost->settings->anchorPrefix.$linkToAnchor;
        $linkObj->URL($url);
        return true;
      }
    }
  }

  function SetupSerialize(array &$props) {
    parent::SetupSerialize($props);
    $props['str'][] = 'page';
  }

  function Parse() {
    if ($format = &$this->settings->format) {
      parent::Parse();
    } else {
      $this->isBlock = true;

      if ($doc = UWikiDocument::TryParsing($this->raw, $this->settings, $this->DefaultStyle())) {
        $this->children[] = $doc->root;
      } else {
        $error = $this->children[] = $this->NewElement('Uinclude_WrongPage');
        $raw = mb_substr($this->raw, 0, 15);
        mb_strlen($this->raw) > 15 and $raw .= '...';
        $error->format = array(ucfirst(UWikiDocument::MarkupNameFrom($this->className)), $raw);
      }
    }
  }

  function Execute($format, $params) {
      $pageFormat = self::IsEmptyStr( $params['format'] ) ? $this->DefaultStyle() : $params['format'];

    $page = &$params['page'];
    if (self::IsEmptyStr($page)) {
      $this->page = null;
      $raw = $format->raw;
    } else {
      $this->page = $page;
      $raw = $this->pager->ReadPage($params['page'], $pageFormat);
    }

    $this->isBlock = $format->blockExpected;
    $format->appendMe = true;

    $format->userData['tree'] = &$params['tree'];

    if ($raw === null or $raw === false) {
      $error = $this->children[] = $this->NewElement('Uinclude_WrongPage');
      $error->format = array( ucfirst($format->current['markupName']), $params['page'] );
    } else {
      $this->IncludePage($raw, $format, $pageFormat, $params);
    }
  }

  function IncludePage(&$raw, $format, $pageFormat, &$params) {
    if (empty( $params['isolate'] )) {
      $format->raw = &$raw;
      $format->root = $this;

      $format->IsEmpty() and $format->AddFormat($pageFormat);

        $nextSettings = $format->NextFormatSettings();
        $nextSettings->pager = clone $nextSettings->pager;
        $nextSettings->AlterPathsBy( str_replace('\\', '/', $this->page).".$pageFormat" );

      if ($format->remaining[0]['markupName'] === 'wacko') {
        $raw .= "\n{{IncludeCorrectHeaders}}";
      } else {
        $format->HookAfterNextFormat($this, 'CorrectEmbedded');
      }
    } else {
      $detachedSettings = $format->topmostDoc->settings->FullClone();
      $detachedSettings->format = null;
      $detachedSettings->LinkTo(new UWikiSettings);

      if ($doc = UWikiDocument::TryParsing($raw, $detachedSettings, $pageFormat)) {
        $this->children[] = $doc->root;
        $this->AfterEmbedding($doc);
      }
    }
  }

    function CorrectEmbedded($calledFormat, $format) {
      $doc = $calledFormat['doc'];
      $this->AfterEmbedding($doc);

      $prevHeading = self::NearestHeadingToSelfIn($format, true);
      $baseLevel = $prevHeading ? $prevHeading->level : 1;
      $baseLevel = self::TreeArgToLevel($format->userData['tree'], $baseLevel);

      $topmostLevel = 256;

        foreach ($doc->settings->headings as $obj) {
          $obj->level += $baseLevel;
          $topmostLevel = min($topmostLevel, $obj->level);
        }

      if ($topmostLevel !== 256) {
        if ($nextHeading = self::NearestHeadingToSelfIn($format, false)
            and $nextHeading->level > $topmostLevel) {
          $error = $this->NewElement('Uinclude_NoProperHeading');
          $error->format = array($topmostLevel, $nextHeading->level);
          $this->children = array($error);
        }
      }
    }

      static function TreeArgToLevel($tree, $baseLevel) {
        static $aliases = array('top' => '-1', 'topmost' => '-999');

        $aliasedTo = &$aliases[$tree];
        isset($aliasedTo) and $tree = $aliasedTo;

        if (is_numeric($tree)) {
          $baseLevel = $tree[0] === '-' ? ($baseLevel + $tree) : $tree;
          $baseLevel < 1 and $baseLevel = 1;
        }

        return $baseLevel;
      }

      static function NearestHeadingToSelfIn($format, $searchBefore) {
          $formatElem = $format->origRoot;

        if ($headings = $formatElem->settings->headings) {
          $selfPos = $formatElem->treePosition;
          $selfIndex = array_pop($selfPos);

          $nearestHeading = null;
          $nearestHeadingIndex = $searchBefore ? 0 : 256;

          foreach ($headings as $heading) {
            if (array_slice($heading->treePosition, 0, -1) === $selfPos) {
              $headingIndex = self::LastIn($heading->treePosition);
              if ($searchBefore ? ($headingIndex < $selfIndex and $headingIndex > $nearestHeadingIndex)
                                : ($headingIndex > $selfIndex and $headingIndex < $nearestHeadingIndex)) {
                $nearestHeadingIndex = $headingIndex;
                $nearestHeading = $heading;
              }
            }
          }

          return $nearestHeading;
        }
      }

    function AfterEmbedding($doc) {
      if ($this->page !== null) {
        // $page is null if called using %%(Include).
        $obj = $doc->GetTitle();
        if (is_object($obj)) {
          $obj->AnchorizeIfNeeds();
          $anchor = $obj->anchor;
        } else {
          $anchor = $this->anchor = $this->NewElement($doc->root->anchorClass);
          $anchor->PointTo($this);
          $anchor->SetUniqueNameFrom($this->page);
          $anchor->isBlock = true;
          $anchor->htmlClasses[] = 'include';
        }

        $this->settings->format->topmostDoc->settings->Included[$this->page] =
          array('anchor' => $anchor, 'doc' => $doc);
      }
    }

  function Priority($format, $params) { return (int) UWikiMaxPriority * 0.7; }

  function AllToHTML() {
    $result = $this->anchor ? $this->anchor->AllToHTML() : '';

    // we'll have 2 children on wrong heading structure: 1st - message,
    // 2nd - included doc's root. We only need the first child whatever it is.
    $root = $this->children[0];
    $endRendering = $root->doc !== $this->doc;
    $endRendering and $root->doc->BeginRenderingInto('html');

      if ($this->isBlock or $root instanceof UWikiFormatErrorMessage) {
        $result .= $root->AllToHTML();
      } elseif ($root->isBlock) {
        $parent = $root->FindInline();
        if ($parent === false) {
          $error = $this->NewElement('Uinclude_OnlyBlockChildren');
          $error->format = array($this->page === null ? '' : ' '.$this->page);
          $result .= $error->AllToHTML();
        } elseif ($parent !== null) {
          $result .= $parent->ChildrenToHTML();
        }
      } else {
        $result .= $root->AllToHTML();
      }

    $endRendering and $root->doc->EndRenderingInto('html', $result);
    return $result;
  }
}

  class Uinclude_WrongPage extends UWikiFormatErrorMessage {
    public $message = '{{include: wrong page';
    public $defaultMessage = 'Cannot {{%s}} %s.';
  }

  class Uinclude_NoProperHeading extends UWikiFormatErrorMessage {
    public $message = '{{include: no proper heading';
    public $defaultMessage = '{{Include}} must be followed by a heading with level not more (level 1 = topmost) than topmost heading of the included document. However, topmost embedded heading had level %d while heading following {{Include}} was of level %d (must be more than/equal to embedded\'s topmost).';
  }

  class Uinclude_OnlyBlockChildren extends UWikiFormatErrorMessage {
    public $message = '{{include: only blocks present';
    public $defaultMessage = 'Cannot inline {{Include}}d document%s because it only has block elements.';
  }

class Uincludecorrectheaders_Root extends UWikiBaseAction {
  public $runsImmediately = true;

  function Execute($format, $params) {
    $actionElement = $format->settings->format->origRoot;
    $actionElement->children[0]->CorrectEmbedded($format->settings->format->current, $format->settings->format);
  }
}

class Urundoc_Root extends Uinclude_Root {
  public $runsImmediately = true;

  function IncludePage(&$raw, $format, $pageFormat, &$params) {
    if (UWikiDocument::TryParsing($raw, $this->settings, $pageFormat)) {
      $format->appendMe = false;
    } else {
      $this->children = array( $this->FormatterError($this->page) );
    }
  }
}
