<?php

abstract class EUWikiSerialization extends EUverseWiki { }

  class EUWikiSerialize extends EUWikiSerialization { }
  class EUWikiUnserialize extends EUWikiSerialization { }

abstract class UWikiAbstractSerializer {
  // these values are used to check input 'count' fields of following entity
  // (array/string) - if they're exceeded the stream is probably corrupt and
  // better exit now than cause PHP to flood all available memory.
  const MaxStringLength = 0x200000;   // 2 MiB
  const MaxArrayCount = 0xFFFF;       // 65535 members; may be more

  const EndByte = 0xB6;               // '¶'

  const BoolTrue  = 84;               // 'T'
  const BoolFalse = 102;              // 'f'
  // for Read/WriteNestedArray():
  const ArrayType = 65;               // 'A'
  const OtherType = 111;              // 'o'

  const IdBitsMask = 0x3FFFFFFF;
  const NewIdChecksum = 0x40000000;   // '@'
  const RepeatedIdChecksum = 0x80000000;

  // flags:
  // Un/serialization generally don't require the handle to support fseek().
  // However, when any of WholeStreamFlags are used it must support it; additionally,
  // when serializing fread() must also work (i.e. the stream must be in read/write mode).
  const WholeStreamFlags = 7;         // GZ, CRC32, Filter
  const CRC32 = 1;
  const GZ = 2;
  const Filter = 4;                   // requires $this->filterCallback to be set.

  static $stringLengths = array('C', 'v', 'V', 'V');
  static $packedLengths = array('V' => 4, 'v' => 2, 'C' => 1);

  // NOTE: do not access $handler directly unless you're doing something impacting
  //       sane performance (which is hard to imagine unless each doc node does that
  //       too) - use methods (Pack(), Read(), etc.) instead.
  public $handle, $startOffset;
  public $flags = 0;
  public $filterCallback;             // function (&$buf); return value is unused.

  // function (UWikiAbstractSerializer $ser, $type, $message); return true to throw an exception.
  public $warningCallback;
  // types: 'newer doc version', 'unknown title type', 'bool value', 'bool prop value',
  //        'not nul', 'id count', 'id checksum', 'crc32', 'end byte', 'data length',
  //        'wrong nested array item type' (by ReadNestedArray()), 'stack left wrong doc',
  //        'different settings class', 'other'.
  public $warnings = array();

  protected $ids = array();           // 'id' => UWikiBaseELement
  protected $origHandle;              // when one of WholeStreamFlags is used.

  function __construct($handle) {
    if (!is_resource($handle)) {
      $this->Error(get_class($this).'::__construct() expects a resource $handle, '.gettype($handle).' given');
    }

    $this->handle = $handle;
    $this->startOffset = ftell($handle);
  }

  abstract function Error($msg);

  function Warning($type, $message) {
    $streamOffset = ftell($this->handle);
    $dataOffset = $streamOffset - $this->startOffset;
    $this->warnings[] = compact('type', 'message', 'streamOffset', 'dataOffset');

    if ($func = $this->warningCallback) {
      if (UWikiDocument::Call($func, array($this, $type, $message))) {
        $this->Error("problem at offset $dataOffset ($streamOffset in the stream) - [$type] $message");
      }
    }
  }

  abstract function Finish();

  function SwitchTo($handle, $startOffset = 0) {
    if (is_array($handle)) { list($handle, $startOffset) = $handle; }

    $old = $this->handle;
    $this->origHandle = array($this->handle, $this->startOffset);
    $this->handle = $handle;
    $this->startOffset = $startOffset;
    return $old;
  }

    static function NewMemStream() { return fopen('php://temp', 'w+b'); }

  function Filter(&$buf) {
    $func = $this->filterCallback;

    if (UWikiDocument::CanCall($func)) {
      UWikiDocument::Call($func, array(&$buf));

      if (!strlen($buf)) {
        $this->Error('filter callback returned an empty buffer');
      }
    } else {
      $this->Error('the stream needs to be ran through a filter but no'.
                   get_class($this).'->$filterCallback function is set');
    }
  }

  function Call($callback, $args, $obj = null) {
    switch ($callback[0]) {
    case '$':   $callback[0] = $obj; break;
    case $this: $args[0] === $this and array_shift($args); break;
    }

    return UWikiDocument::Call($callback, $args);
  }

  // debug method:
  function dbg() { echo ftell($this->handle); exit; }
}

  class UWikiSerializer extends UWikiAbstractSerializer {
    public $variableHdrPos;
    public $gzLevel = 3;              // 0 (no compression) - 9 (max); only when GZ flag is set.

    function __construct($handle, $flags = 0) {
      parent::__construct($handle);
      $this->flags = $flags;
    }

    function Error($msg) {
      throw new EUWikiSerialize("Cannot serialize: $msg.");
    }

      function WriteError($expectedLength, $readLength) {
        $this->Error('error writing '.strlen($buf)." bytes - $count written");
      }

    function WriteHeader() {
      $this->Pack('V', $this->flags);
      $this->variableHdrPos = ftell($this->handle);

      if ($this->flags & self::WholeStreamFlags) {
        $this->Pack('V', 0);  // placeholder for data length.

        if ($this->flags & self::CRC32) { $this->Pack('V', 0); }

        if ($this->flags & (self::GZ | self::Filter)) {
          $this->SwitchTo( self::NewMemStream() );
        }
      }
    }

      function Finish() {
        $dataLength = ftell($this->handle) - $this->startOffset + 7;
        $this->Pack('vVC', count($this->ids), $dataLength, self::EndByte);

        if ($this->flags & self::WholeStreamFlags) {
          if ($this->flags & (self::GZ | self::Filter)) {
            $plain = $this->SwitchTo($this->origHandle);
            $buf = stream_get_contents($plain, -1, 0);
            fclose($plain);

              if ($this->flags & self::Filter) { $this->Filter($buf); }

              if ($this->flags & self::GZ) {
                $buf = gzcompress($buf, $this->gzLevel);
                if (!strlen($buf)) { $this->Error('gzcompress() has failed'); }
              }

            $this->Write($buf);
            $buf = null;
          }

          $length = ftell($this->handle) - $this->variableHdrPos - 4;

            fseek($this->handle, $this->variableHdrPos);
            $this->Pack('V', $length);

          if ($this->flags & self::CRC32) {
            fseek($this->handle, +4, SEEK_CUR);   // to skip CRC32 placeholder DWord.
            $buf = fread($this->handle, $length - 4);

              if (strlen($buf) != $length - 4) {
                $this->Error('error writing CRC32 checksum - reading of previous'.
                             ' buffer failed ('.strlen($buf).' bytes read, '.$length.
                             ' required); the stream might no support fseek() or is in write-only mode');
              }

            fseek($this->handle, $this->variableHdrPos + 4);
            $this->Pack('V', crc32($buf));
            $buf = null;
          }
        }
      }

    function Write($buf) {
      $count = fwrite($this->handle, $buf, strlen($buf));
      $count == strlen($buf) or $this->WriteError(strlen($buf), $count);
    }

    function Pack($fmt, $value_1) {
      $args = func_get_args();
      $this->Write( call_user_func_array('pack', $args) );
    }

    function WriteString($str, $maxBytes = 4) {
      $str = (string) $str;
      $this->Pack(self::$stringLengths[$maxBytes - 1], strlen($str));
      $str === '' or $this->Write($str);
    }

      function WriteBool($trueFalse) {
        $this->Pack('C', $trueFalse ? self::BoolTrue : self::BoolFalse);
      }

      function WriteArray(array &$array = null) {
        $buf = $array ? serialize($array) : '';
        $this->WriteString($buf);
      }

      // $serializeCallback = function (UWikiSerializer $ser, &$item); result is unused.
      function WriteNestedArray(array $array = null, $serializeCallback, array $options = array()) {
        $options += array('isNested' => true, 'withKeys' => false);

        $this->Pack('v', count($array));
        if ($array) {
          foreach ($array as $key => &$item) {
            $options['withKeys'] and $this->WriteString($key);

            if (is_array($item) and $options['isNested']) {
              $this->Pack('C', self::ArrayType);
              $this->WriteNestedArray($item, $serializeCallback, $options);
            } else {
              $options['isNested'] and $this->Pack('C', self::OtherType);
              $this->Call($serializeCallback, array($this, $item), $item);
            }
          }
        }
      }

        function WriteArrayUsing($serializeCallback, array $array = null) {
          $this->WriteNestedArray($array, $serializeCallback, array('isNested' => false));
        }

        function WriteNestedHash(array $array = null, $serializeCallback) {
          $this->WriteNestedArray($array, $serializeCallback, array('withKeys' => true));
        }

        function WriteHashUsing($serializeCallback, array $array = null) {
          $options = array('isNested' => false, 'withKeys' => true);
          $this->WriteNestedArray($array, $serializeCallback, $options);
        }

    // $serializeCallback = function (UWikiSerializer $ser, $obj); result is unused.
    function WriteInstance($obj = null, $serializeCallback) {
      if (!$obj) {
        $this->Write("\0\0\0\0");
      } else {
        $id = &$obj->_SerializationID_;
        if (!isset($id)) {
          $id = count($this->ids);
          $id === 0 and $id = 1;
          while (isset( $this->ids[$id] )) { ++$id; }
        }

        if (empty( $this->ids[$id] )) {
          $this->ids[$id] = $obj;

          $class = get_class($obj);
          $this->Pack('VCa*', $id | self::NewIdChecksum, strlen($class), $class);

          $this->Call($serializeCallback, array($this, $obj), $obj);
          return true;
        } else {
          $this->Pack('V', $id | self::RepeatedIdChecksum);
        }
      }
    }

      // $elements must belong to the same environment (Settings) or ID collision might occur.
      function WriteElement(UWikiBaseElement $element = null) {
        $isNew = $this->WriteInstance($element, array($element, 'SerializeTo'));
        if ($isNew) {
          $this->Pack('v', count($element->children));
          foreach ($element->children as $child) { $this->WriteElement($child); }
        }
      }

      function WriteDocument(UWikiDocument $doc = null) {
        $this->WriteInstance($doc, array($doc, 'WriteTo'));
      }

      function WriteSettings(UWikiSettings $settings = null) {
        $this->WriteInstance($settings, array($settings, 'SerializeTo'));
      }

    function WriteSupportedType($value) {
      if (is_string($value)) {
        $this->Write('s');
        $this->WriteString($value);
      } elseif (is_int($value)) {
        $this->Write('i');
        $this->Pack('V', $value);
      } elseif (is_bool($value)) {
        $this->Write('b');
        $this->WriteBool($value);
      } elseif (is_array($value)) {
        $this->Write('a');
        $this->WriteArray($value);
      } elseif (is_object($value)) {
        if ($value instanceof UWikiBaseElement) {
          $this->Write('e');
          $this->WriteElement($value);
        } elseif ($value instanceof UWikiDocument) {
          $this->Write('d');
          $this->WriteDocument($value);
        } else {
          $unsupported = true;
        }
      } elseif (is_float($value)) {
        $this->Write('f');
        $this->Pack('f', $value);
      } else {
        $unsupported = true;
      }

      if (!empty($unsupported)) {
        $this->Error('cannot write value because its type ('.gettype($value).
                     (is_object($value) ? ', class '.get_class($value) : '').
                     'is unsupported');
      }
    }
  }

  class UWikiUnserializer extends UWikiSerializer {
    protected $docs = array();

    function CurrentDoc(UWikiDocument $doc) {
      array_unshift($this->docs, $doc);
    }

    function LeaveDoc(UWikiDocument $expectedToBeLeft = null) {
      $doc = array_shift($this->docs);
      if ($expectedToBeLeft and $doc !== $expectedToBeLeft) {
        $this->Warning('stack left wrong doc', 'expected to leave document #'.$expectedToBeLeft->ID().' but actually left #'.$doc->ID());
      }
    }

    function Error($msg) {
      throw new EUWikiUnserialize(sprintf('Cannot unserialize @%X stream offset (%X data): %s.',
                                          ftell($this->handle), ftell($this->handle) - $this->startOffset, $msg));
    }

      function EnsureNUL() {
        $byte = $this->Read(1);
        $byte === "\0" or $this->Warning('not nul', 'expected a NUL byte but got 0x'.dechex(ord($byte)));
      }

      function PrematureEOF($expectedLength, $readLength) {
        $this->Error("error reading $expectedLength bytes from stream, $readLength were read - stream might have ended prematurely");
      }

      function TooLarge($got, $max) {
        $this->Error("too large length to read - got $got but max is $max; this usually indicates that the input stream is corrupted");
      }

    function ReadHeader() {
      $this->flags = $this->UnpackOne('V');

      if ($this->flags & self::WholeStreamFlags) {
        $length = $this->UnpackOne('V');

        if ($this->flags & self::CRC32) {
          $hash = $this->UnpackOne('V');

          $buf = $this->Read($length - 4);
          if (crc32($buf) !== $hash) {
            $this->Warning('crc32', sprintf('CRC32 checksum differs for data %d bytes long:'.
                                            ' checsum must be %8X, got %8X', $length - 4,
                                            $hash, crc32($buf)));
          }

          $buf = null;
          fseek($this->handle, -1 * $length + 4, SEEK_CUR);
        }

        if ($this->flags & (self::GZ | self::Filter)) {
          $old = $this->SwitchTo( self::NewMemStream() );
          $buf = fread($old, $length);

            if ($this->flags & self::GZ) {
              $buf = gzuncompress($buf);
              if (!strlen($buf)) { $this->Error('gzuncompress() has failed'); }
            }

            if ($this->flags & self::Filter) { $this->Filter($buf); }

          $this->Write($buf);
          rewind($this->handle);
          $buf = null;
        }
      }
    }

      function Finish() {
        $idCount = $this->UnpackOne('v');
        if ($idCount !== count($this->ids)) {
          $this->Warning('id count', 'ID count after unserializing differs ('.count($this->ids).') from what\'s'.
                                     " written was when the document was serialized ($idCount)");
        }

        $tail = $this->Unpack('Vlength/Ccheck', 5);

          if ($tail['check'] !== self::EndByte) {
            $this->Warning('end byte', 'end byte differs: 0x'.dechex(self::EndByte).
                           ' expected but got 0x'.dechex($tail['check']));
          }

          $length = ftell($this->handle) - $this->startOffset;
          if ($tail['length'] !== $length) {
            $this->Warning('data length', "input stream expected to be $length bytes".
                                          " long but only $tail[length] were read");
          }

        if ($this->flags & (self::GZ | self::Filter)) {
          $plain = $this->SwitchTo($this->origHandle);
          fclose($plain);
        }
      }

    function &Read($length) {
      $buf = fread($this->handle, $length);
      isset($buf[$length - 1]) or $this->PrematureEOF($length, strlen($buf));
      return $buf;
    }

    function &Unpack($fmt, $length) {
      $result = unpack($fmt, $this->Read($length));
      if (!$result) { $this->Error("error unpacking $length bytes as '$fmt'"); }
      return $result;
    }

    function UnpackOne($fmt, $length = null) {
      $length or $length = self::$packedLengths[$fmt];
      $data = $this->Unpack($fmt, $length);

      $result = array_pop($data);
      if (!isset($data)) {
        $this->Error("error unpacking single value (got none) from $length bytes as '$fmt'");
      } elseif (!empty($data)) {
        $this->Error('error unpacking single value (got '.count($data).") from $length bytes as '$fmt'");
      }

      return $result;
    }

    function ReadString($maxBytes = 4) {
      $length = unpack(self::$stringLengths[$maxBytes - 1], fread($this->handle, $maxBytes));
      if ($length[1] > self::MaxStringLength) {
        $this->TooLarge($length[1], self::MaxStringLength);
      } elseif ($length[1] > 0) {
        $result = fread($this->handle, $length[1]);
        if (!isset( $result[$length[1] - 1] )) { $this->PrematureEOF($length[1], strlen($result)); }
      } else {
        $result = '';
      }

      return $result;
    }

      function ReadBool() {
        $result = $this->UnpackOne('C');
        if ($result === self::BoolTrue or $result === self::BoolFalse) {
          return $result === self::BoolTrue;
        } else {
          $this->Warning('bool value', "wrong bool value - got $byte but expected either ".
                                       self::BoolTrue.' (for true) or '.self::BoolFalse.' (for false)');
          return $result !== 0;
        }
      }

      function ReadArray() {
        $result = $this->ReadString();
        if ($result === '') {
          $result = array();
        } else {
          $result = unserialize($result);
          if (!is_array($result)) {
            $this->Error('expected a serialized array but after unserialization got a '.gettype($result));
          }
        }

        return $result;
      }

      // $unserializeCallback = function (UWikiUnserializer $ser,); returns unserialized item.
      function ReadNestedArray($unserializeCallback, array $options = array()) {
        $options += array('isNested' => true, 'withKeys' => false);

        $result = array();

        $count = unpack('v', fread($this->handle, 2));
        $count or $this->PrematureEOF(2, 0);
        if ($count[1] > self::MaxArrayCount) {
          $this->TooLarge($count[1], self::MaxArrayCount);
        }

          $callbackArgs = $unserializeCallback[0] === $this ? array() : array($this);

          for (; --$count[1] >= 0; ) {
            $key = $options['withKeys'] ? $this->ReadString() : count($result);

            $isArray = $options['isNested'] ? $this->UnpackOne('C') : self::OtherType;

            if ($isArray !== self::ArrayType and $isArray !== self::OtherType) {
              $this->Warning('wrong nested array item type', 'ReadNestedArray()'.
                             ' expected either '.self::ArrayType.' (is array)'.
                             ' or '.self::OtherType.' (is other) byte but got '.$isArray);
            } elseif ($isArray === self::ArrayType) {
              $result[$key] = $this->ReadNestedArray($unserializeCallback, $options);
            } else {
              $result[$key] = UWikiDocument::Call($unserializeCallback, $callbackArgs);
            }
          }

        return $result;
      }

        function ReadArrayUsing($unserializeCallback) {
          return $this->ReadNestedArray($unserializeCallback, array('isNested' => false));
        }

        function ReadNestedHash($unserializeCallback) {
          return $this->ReadNestedArray($unserializeCallback, array('withKeys' => true));
        }

        function ReadHashUsing($unserializeCallback) {
          $options = array('isNested' => false, 'withKeys' => true);
          return $this->ReadNestedArray($unserializeCallback, $options);
        }

    // $unserializeCallback = function (UWikiUnserializer $ser, &$newObj, $className)
    // $newObj must be set to the instance of new (unserialized) object as
    // soon as possible so it's registered with assigned ID.
    function ReadInstance($unserializeCallback) {
      $id = unpack('V', fread($this->handle, 4));
      $id = $id[1];
      is_int($id) or $this->PrematureEOF(4, 0);

      if ($id === 0) {
        return null;
      } else {
        $isNew = (bool) (self::NewIdChecksum & $id);
        $id &= self::IdBitsMask;

          if ($isNew !== empty( $this->ids[$id] )) {
            $this->Warning('id checksum', 'wrong ID checksum - object is expected to be '.
                           (empty( $this->ids[$id] ) ? 'new' : 'not new').' but stream indicates the opposite');
          }

        if (empty( $this->ids[$id] )) {
          $length = ord( fread($this->handle, 1) );
          $class = fread($this->handle, $length);
          isset( $class[$length - 1] ) or $this->PrematureEOF($length, strlen($class));

          $newObj = &$this->ids[$id];
          if (is_array($unserializeCallback) and count($unserializeCallback) === 2) {
            call_user_func_array($unserializeCallback, array($this, &$newObj, $class));
          } else {
            UWikiDocument::Call($unserializeCallback, array($this, &$newObj, $class));
          }
        }

        return $this->ids[$id];
      }
    }

      function ReadElement() {
        // could use ReadInstanceOf() but elements are read hungreds or thousands times per
        // doc so creating call stack alone for calling it is costy.
        if ($element = $this->ReadInstance( array(__CLASS__, 'ReadElementCallback') )
            and !($element instanceof UWikiBaseElement)) {
          $got = $obj ? get_class($element) : 'NULL';
          $this->Error("unserialized element is expected to be of class UWikiBaseElement but got $got");
        }

        return $element;
      }

        static function ReadElementCallback(UWikiUnserializer $ser, &$element, $class) {
          if (!class_exists($class)) {
            UWikiDocument::PreloadMarkup( UWikiDocument::MarkupNameFrom($class) );
          }
          if (!class_exists($class)) {
            $ser->Error("element class $class is undefined");
          }

          if (empty($ser->docs)) {
            $ser->Error('no current document to bind new element to');
          }

          $element = $ser->docs[0]->NewElement($class);
          $element->UnserializeFrom($ser);

          $count = unpack('v', fread($ser->handle, 2));
          is_int($count[1]) or $ser->PrematureEOF(2, 0);
          for (; --$count[1] >= 0; ) { $element->children[] = $ser->ReadElement(); }
        }

      function ReadDocument(array $options = array()) {
        return $this->ReadInstanceOf('UWikiDocument', array('UWikiDocument', 'ReadFrom', $options));
      }

      function ReadSettings() {
        return $this->ReadInstanceOf('UWikiSettings', array($this, 'CreateAndReadSettingsCallback'));
      }

        function CreateAndReadSettingsCallback(UWikiUnserializer $ser = null, &$newObj, $class) {
          $newObj = new $class;
          $newObj->UnserializeFrom($ser, $newObj, $class);
        }

      function ReadSettingsInto(UWikiSettings $settings) {
        $this->ReadInstanceOf('UWikiSettings', array($this, 'ReadSettingsCallback', $settings));
      }

        function ReadSettingsCallback(UWikiUnserializer $ser = null, &$obj, $class, $settings) {
          if (! $settings instanceof $class) {
            $this->Warning('different settings class', 'settings objects to read into is of class'.
                           get_class($settings)." but stream contains an instance of $class which".
                           ' has different inheritance tree and might be incompatible');
          }

          $settings->UnserializeFrom($ser, $obj, $class);
        }

    function ReadInstanceOf($expectedClass, $unserializeCallback) {
      $obj = $this->ReadInstance($unserializeCallback);

        if ($obj and !($obj instanceof $expectedClass)) {
          $got = $obj ? get_class($obj) : 'NULL';
          $this->Error("unserialized object is expected to be of class $expectedClass but got $got");
        }

      return $obj;
    }

    function ReadSupportedType() {
      switch ($type = $this->Read(1)) {
      case 's': return $this->ReadString();
      case 'i': return $this->UnpackOne('V');
      case 'b': return $this->ReadBool();
      case 'a': return $this->ReadArray();
      case 'e': return $this->ReadElement();
      case 'd': return $this->ReadDocument();
      case 'f': return $this->UnpackOne('f');
      default:  $this->Error("cannot read value - cannot recognize its type ('$type')");
      }
    }
  }
