<?php
  // don't use chars that are special in PCRE.
  define('UWikiFormatParamSeparator', ',');  // this can be space (will be like original WackoWiki).
  define('UWikiFormattersSeparator', ';');

class UWikiStyle {
  public $settings;

  function __construct(UWikiSettings $settings) {
    $this->settings = $settings;
  }

  // "style" is as simple as "(wacko)".
  function ExtractFrom(&$text, $offset = 0, $allowMultiple = false) {
    $regexp = '/^\((' .$this->StylePattern($allowMultiple). ')\)\s*/xu';

    if (@$text[$offset] == '(' and preg_match($regexp, substr($text, $offset), $matches)) {
      $text = mb_substr($text, $offset + mb_strlen($matches[0]));
      return $allowMultiple ? explode(' ', $matches[1]) : $matches[1];
    }
  }

    function StylePattern($allowMultiple = false) {
      $msl = $this->settings->maxStyleNameLength - 1;
      $wordCh = $this->settings->wordCharacters;
      $pattern = "[a-zA-Z$wordCh][0-9_$wordCh]{0,$msl}";
      return $allowMultiple ? "($pattern)([ ]+$pattern)*" : $pattern;
    }

  function ExtractMultipleFrom(&$text, $offset = 0) {
    $styles = $this->ExtractFrom($text, $offset, true);
    return $styles ? $styles : array();
  }

    // needs /x modifier.
    // $matches: [1] - param name, if quoted; [2] - name otherwise; [3] - value, if quoted; [4] - value otherwise.
    function ParamPattern() {
      return  '
                  (?:
                      ( (?:" [^"]* ")+ )
                    | ( [^='.UWikiFormattersSeparator.UWikiFormatParamSeparator.')]+ )
                  )
                (?:
                  \s* = \s*
                  (?:
                      ( (?:" [^"]* ")+ )
                    | ( [^='.UWikiFormattersSeparator.UWikiFormatParamSeparator.')]+ )
                  )
                )?
              ';
    }

    // needs /x modifier.
    // $matches: [1] - style name, [2] - parameter string (if any).
    function FormatPattern() {
      $paramPattern = $this->ParamPattern();
      return  '
                ('.$this->StylePattern().')
                :?
                ( (?:
                  \s+
                  '.$paramPattern.'
                  (?:
                    \s* ['.UWikiFormatParamSeparator.'] \s*  # wrapped in [] so that even space will be matched.
                    '.$paramPattern.'
                  )*
                )? )
              ';
    }

  // "format" is several format strings, each with possible format parameters. E.g.: (if out=html; wacko)
  // returns array of array( null => 'style'[, param => value, ...] )
  function &ExtractFormatFrom(&$text, $offset = 0) {
    $formats = array();

      $pSepar = UWikiFormatParamSeparator;
      $fSepar = UWikiFormattersSeparator;

      $paramPattern = $this->ParamPattern();
      $stylePattern = $this->StylePattern();
      $formatPattern = $this->FormatPattern();

    $regexp =  "/^
                  \(
                    (?:
                      ($formatPattern)
                      ( (?:
                        \s* $fSepar \s+
                        ($formatPattern)
                      )* )
                    | ($stylePattern)
                      ( (?:
                        \s* [$pSepar] \s*
                        ($stylePattern)
                      )* )
                    )
                  \) \s*
                /xu";

    if (@$text[$offset] == '(' and preg_match($regexp, substr($text, $offset), $genMatches)) {
      if ($altSyntax = &$genMatches[24]) {
        // as requested by Freeman: %%(mirror, html) instead of %%(mirror; html).
        $formats = explode($pSepar, $altSyntax.$genMatches[25]);
        foreach ($formats as &$format) { $format = array(null => trim($format)); }
      } elseif (preg_match_all("/$fSepar\s+ $formatPattern (?=\s*$fSepar)/xu",
                         "$fSepar $genMatches[1]$genMatches[12]$fSepar", $fmtMatches, PREG_SET_ORDER)) {
        foreach ($fmtMatches as &$format) {
          list(, $style, $paramStr) = $format;
          $format = array(null => $style);

            if (preg_match_all("/\s*[$pSepar]\s* $paramPattern/xu", "$pSepar $paramStr", $paramMatches)) {
              foreach ($paramMatches[0] as $i => $oneParam) {
                $name = trim( $this->QuotedOrPlain( $paramMatches[1][$i], $paramMatches[2][$i] ) );
                $value = $this->QuotedOrPlain( $paramMatches[3][$i], $paramMatches[4][$i] );

                $name === '' or $format[$name] = ($value === null ? true : trim($value));
              }
            }

          $formats[] = $format;
        }
      }

      $text = substr( $text, $offset + strlen($genMatches[0]) );
    }

    return $formats;
  }

    // returns null if neither Plain nor Quoted were passed.
    // Quoted === '""' isn't a null, just an empty string: %%(fmt param="")
    function QuotedOrPlain($valueIfQuoted, $valueIfPlain) {
      $value = null;
      $valueIfPlain === '' or $value = $valueIfPlain;
      $valueIfQuoted === '' or $value = str_replace('""', '"', substr($valueIfQuoted, 1, -1));
      return $value;
    }
}

class UWikiFormat {
  public $raw, $root, $settings;
  public $origRoot, $origDoc, $topmostDoc;
  public $renderingIn;           // null if output format is unknown (e.g. during parse phase); otherwise 'html', etc.
  public $userData = array();    // field for storing arbitrary data; isn't used by this class.

  // Set $onError before adding formats. Format that will be used in place of inexistent/wrong/etc. formatters.
  public $onError = array(null => UWikiTextMarkup);
  public $hooks = array();       // array of array(formatIndex => function ($calledFormat, UWikiFormat))

  public $remaining = array();   // for its structure see AddFormats().
  public $current;
  public $blockExpected;

  public $appendMe;
  public $stop = false;  // skips remaining formats.

  function __construct(UWikiBaseElement $root = null, $formatsOrRaw = null, $raw = null) {
    if (!is_array($formatsOrRaw) and $raw === null) {
      $raw = $formatsOrRaw;
      $formatsOrRaw = null;
    }

    $this->raw = &$raw;
    $root and $this->ReattachTo($root);

    $formatsOrRaw and $this->AddFormats($formatsOrRaw);
  }

    function ReattachTo(UWikiBaseElement $root) {
      $this->origRoot = $this->root = $root;
      $this->origDoc = $root->doc;
      $this->topmostDoc = $root->settings->format ? $root->settings->format->topmostDoc : $root->doc;
      $this->settings = &$root->settings;
    }

  function AddFormat($formatter, $params = array()) {
    $this->AddFormats( array(array(null => $formatter) + $params) );
  }

  function AddFormats($formats) {
    foreach ($formats as $format) {
      $format = $this->MakeFormatFrom($format);
      $format and $this->remaining[] = $format;
    }
  }

    function MakeFormatFrom($format) {
      $markupName = $format[null];
      if ($markupName) {
        unset( $format[null] );

        if (!$this->root->IsStyleDisabled($markupName)) {
          $markupName = $this->root->RealStyleBy($markupName);
          try {
            $doc = $this->origDoc->NewLinkedDocument('', $markupName);

              $inPlaceOfError = @$format['inPlaceOfError'];
              if ($inPlaceOfError and ($doc->root instanceof UWikiStringMessage)) {
                $doc->root->format = array( ucfirst($inPlaceOfError) );
              }

            return array('doc' => $doc, 'inPlaceOfError' => $inPlaceOfError, 'hookDocOnRender' => true,
                         'markupName' => $markupName, 'params' => $doc->root->PrepareParams($format));
          } catch (EUverseWiki $e) {
          }
        }

        if ($markupName !== $this->root->RealStyleBy( $this->onError[null] )) {
          $onError = $this->onError + $format;
          $onError['inPlaceOfError'] = $markupName;
          return $this->MakeFormatFrom($onError);
        }
      }
    }

  // note that if all remaining formats are static it returns null but not an empty UWikiFormat.
  function SplitDynamic() {
    foreach ($this->remaining as $i => $format) {
      if ($this->CallOn($i, 'IsDynamic')) {
        $dynaFormat = new self($this->origRoot, array(), $this->raw);
        $dynaFormat->remaining = array_splice($this->remaining, $i);
        return $dynaFormat;
      }
    }
  }

    // Make sure not to creat eloops when one format waits for another to determine
    // if it's dynamic or not and that in turns asks the first format the same question.
    // To avoid this set $start after your format's index in this formatting chain.
    function HasDynamic($start = 1) {
      for (; isset( $this->remaining[$start] ); ++$start) {
        if ($this->CallOn($start, 'IsDynamic')) { return true; }
      }
    }

  function RunImmediateFormats() {
    foreach ($this->remaining as $i => $format) {
      if ($format['doc']->root->runsImmediately) {
        $this->Apply($format);
        $this->remaining[$i] = null;
      } else {
        $this->CallOn($i, 'WasFound');
      }
    }

    $this->stop = false;
    $this->remaining = array_filter($this->remaining);
  }

  // returns true if at least one formatter has run ok.
  function ApplyAll() {
    $oneOK = false;

      while (!$this->IsEmpty() and !$this->stop) {
        $errorFormat = $this->ApplyOne();
        $oneOK |= $errorFormat === null;
      }

    return $oneOK;
  }

    function ApplyOne() {
      $hooks = array_shift($this->hooks);
      $format = array_shift($this->remaining);

      $status = $this->Apply($format);

      UWikiDocument::CallAll($hooks, array($format, $this));
      return $status;
    }

  // returns null on success or name of format instead of which $this->onError was invoked.
  function Apply($format) {
      if (!is_array($format)) { return; }

    $this->appendMe = true;
    $this->blockExpected = $this->root->isBlock;
    $this->current = $format;
    $joint = $this->root;

      $formatter = $format['doc'];
      $formatter->SetSource($this->raw);

      $treePos = $joint->treePosition;
      $treePos[] = count($joint->children) + 1;
      $formatter->root->treePosition = $treePos;

      $formatter->settings->format = $this;
      $formatter->settings->LinkTo($this->settings);
      $formatter->Parse();

      if ($this->appendMe) {
        $joint->children[] = $formatter->root;

        if ($format['hookDocOnRender']) {
          $this->origDoc->beforeRenderingThisHooks[] = array(__CLASS__, 'DocOnRenderHook',
                                                             $this->origDoc, $formatter);
        }
      }

    return $format['inPlaceOfError'];
  }

    static function DocOnRenderHook($format, UWikiDocument $doc, UWikiDocument $formatter) {
      $origDoc->afterRenderingThisHooks[] = array($formatter, 'EndRenderingInto');
    }

  function IsEmpty() { return empty($this->remaining); }
  function Clear() { $this->remaining = array(); }
  function Priority() { return (int) $this->CallOn(0, 'Priority'); }

  function CallOn($formatIndex, $method) {
    $format = &$this->remaining[$formatIndex];
    return $format ? $format['doc']->root->$method($this, $format['params']) : null;
  }

  function HookAfterNextFormat($objOrFunc, $method = null) {
    $this->hooks[0][] = func_get_args();
  }

  function NextFormatDoc() {
    if (empty($this->remaining)) {
      throw new EUverseWiki('No remaining formats to return NextFormatDoc() for.');
    } else {
      return $this->remaining[0]['doc'];
    }
  }

    function NextFormatSettings() { return $this->NextFormatDoc()->settings; }
    function NextMarkup() { return $this->NextFormatDoc()->LoadedMarkup(); }

  function SerializeTo(UWikiSerializer $ser) {
    $ser->WriteString($this->raw);
    $ser->WriteElement($this->root);
    $ser->WriteElement($this->origRoot);
    $ser->WriteDocument($this->origDoc);
    $ser->WriteDocument($this->topmostDoc);
    $ser->WriteString($this->renderingIn);
    $ser->WriteArray($this->userData);
    $ser->WriteArray($this->onError);

    $remaining = $this->remaining;
    array_unshift($remaining, $this->current);

      $ser->Pack('v', count($remaining));
      foreach ($remaining as &$format) {
        if ($format === null) {
          $ser->Write("\1");
        } else {
          $ser->Write("\0");

          $ser->WriteDocument($format['doc']);
          $ser->WriteString($format['inPlaceOfError']);
          $ser->WriteString($format['markupName']);
          $ser->WriteArray($format['params']);
        }
      }

    $ser->WriteBool($this->blockExpected);
    $ser->WriteBool($this->appendMe);
    $ser->WriteBool($this->stop);
  }

    static function UnserializeFrom(UWikiUnserializer $ser, &$style, $class) {
      $style = new $class;

      $style->raw = $ser->ReadString();
      $style->root = $ser->ReadElement();
        $style->settings = &$style->root->settings;
      $style->origRoot = $ser->ReadElement();
      $style->origDoc = $ser->ReadDocument();
      $style->topmostDoc = $ser->ReadDocument();
      $style->renderingIn = $ser->ReadString();
      $style->userData = $ser->ReadArray();
      $style->onError = $ser->ReadArray();

      $count = $ser->UnpackOne('v');
      if ($count > UWikiUnserializer::MaxArrayCount) {
        $ser->TooLarge($count > UWikiUnserializer::MaxArrayCount);
      }

        for (; --$count >= 0; ) {
          $isNull = $ser->UnpackOne('C');
          if ($isNull === 1) {
            $style->remaining[] = null;
          } elseif ($isNull === 0) {
            $style->remaining[] = array('doc' => $ser->ReadDocument(),
                                       'inPlaceOfError' => $ser->ReadString(),
                                       'markupName' => $ser->ReadString(),
                                       'params' => $ser->ReadArray());
          } else {
            $style->Warning('other', __CLASS__."->UnserializeFrom() expected either 0 or 1 for 'isNull' but got '$isNull'");
          }
        }

        $style->current = array_shift($style->remaining);

      $style->blockExpected = $ser->ReadBool();
      $style->appendMe = $ser->ReadBool();
      $style->stop = $ser->ReadBool();
    }
}
