<?php

class BPosts {
  static $docs = array();

  // $filter: array('field', <args>[, ...])
  //          'field': dateyear/month/day (1 arg), tag (1 arg), and, or.
  // "or" condition is default for "@date/@date", "and" - for "=tag/=tag".
  // See also Filter class.
  static function By($filter = array()) {
    $filter = BFilter::Normalize($filter);

    $first = array_shift($filter);
    if ($first) {
      $posts = call_user_func_array(array('BIndex', 'SelectFrom'), $first);
    } else {
      $posts = self::All();
    }

    $condition = null;
    foreach ($filter as $criteria) {
      $field = $criteria[0];

      if ($field === 'or' or $field === 'and') {
        $condition = $field;
      } else {
        if (!$condition) {
          $condition = 'and';
          if (substr($filter[0][0], 0, 4) === 'date' and
              substr($field, 0, 4) === 'date') { $condition = 'or'; }
        }

        $index = array_shift($criteria);
        $thisCritPosts = call_user_func_array(array($index, 'SelectFrom'), $criteria);

        if ($condition === 'or') {
          $posts = array_merge($posts, $thisCritPosts);
        } else {
          $posts = array_intersect($posts, $thisCritPosts);
        }
      }
    }

    return array_values($posts);
  }

  static function SortAs($way, &$posts) {
    if (method_exists(__CLASS__, 'SortBy'.$way)) {
      usort($posts, array(__CLASS__, 'SortBy'.$way));
    } else {
      BEvent::Fire('sort posts: '.$way, array(&$posts));
    }
  }

    static function SortByTimeDesc($first, $second) {
      $f = self::DateOf($first);
      $s = self::DateOf($second);
      return ( $f > $s ? -1 : ($f < $s ? +1 : 0) );
    }

  static function All() { return BIndex::SelectFrom('posts'); }
  static function Exists($post) { return BIndex::SelectFrom('posts', $post); }
  static function DateOf($post) { return BIndex::SelectFrom('PostDates', $post); }

  static function TitleOf($post, $asHTML) {
    return self::Get($post, $asHTML ? 'postTitle' : 'textTitle');
  }

  static function NextTo($post) { return self::GetByOffsetTo($post, +1); }
  static function PreviousTo($post) { return self::GetByOffsetTo($post, -1); }
  static function First() { return self::GetByOffsetTo(null, 'first'); }
  static function Last() { return self::GetByOffsetTo(null, 'last'); }
  static function Random() { return self::GetByOffsetTo(null, 'random'); }

    static function GetByOffsetTo($post, $offset) {
      return BIndex::SelectFrom('PostDates', $post, $offset);
    }

  // there can be several posts when URL tail contains SeqNum more than there are posts.
  static function AllMatchingUrlTitle($title) {
    $title = trim($title, '\\/');

    $post = BIndex::SelectFrom('UrlTitles', $title);
    $post or $post = BIndex::SelectFrom('UrlTitles', mb_strtolower($title));
    $post or BEvent::Fire('posts by URL', array($title, &$post));

    return $post ? ((array) $post) : array();
  }

    // e.g.: http://myblog/Some-post's-title/
    static function ByUrlTitle($title) {
      $posts = self::AllMatchingUrlTitle($title);
      return empty($posts) ? null : $posts[0];
    }

  static function CanComment($post) {
    $state = BPosts::Get($post, 'commenting');
    return $state === null ? BConfig::$canCommentPosts : $state;
  }

  static function Get($posts, $field = null) {
    if (is_array($posts)) {
      $result = array();
      foreach ($posts as $post) { $result[] = self::Get($post, $field); }
      return $result;
    } else {
      $vars = BIndex::SelectFrom('PostDocs', $posts);
      if (!$vars) { throw new BException("Post $posts doesn't exist."); }
      isset($field) and $vars = $vars[$field];
      return $vars;
    }
  }

  // returns full canonical URL including domain name.
  static function UrlOf($post) {
    return BConfig::$siteHome.self::RelativeUrlOf($post);
  }

    static function ShortUrlOf($post) {
      if (Permalink::IsTailEnabledFor($post) and $tail = Permalink::TailFrom(self::UrlOf($post))) {
        return BConfig::$siteHome.$tail;
      }
    }

    static function RelativeUrlOf($post) {
      return self::Get($post, 'permalink');
    }

    static function CommentsUrlOf($post) {
      return self::UrlOf($post).'#comments';
    }

  // $info['doc'] is expected to be an instance of UWikiDocument.
  // 'category' and any other keys will be taken into account too if exist.
  static function PermalinkFor($info, &$extraLinks, $post = null) {
    if ($info instanceof UWikiDocument) { $info = array('doc' => $info); }

    if (isset($info['doc']) and !($info['doc'] instanceof UWikiDocument)) {
      throw new BException(__CLASS__.'::PermalinkFor() expects an instance of UWikiDocument'.
                          ' in $info[\'doc\'] but a wrong type/class was given: '.var_export($info['doc'], true));
    }

    return BEvent::FireResult('string', 'post permalink', array(&$info, &$extraLinks, $post));
  }

  static function FormatLinkOf($post, $format) {
    return self::FormatLinkFor(self::Get($post), $format, $post);
  }

    static function FormatLinkFor($info, $format, $post = null) {
      if ($info instanceof UWikiDocument) { $info = array('doc' => $info); }
      BEvent::Fire('format post link', array(&$format, &$info, $post));
      return $format;
    }

  // returns null if document is already short enough.
  static function Shorten($doc, $length, $options = array()) {
    $options += array('fullURL' => null, 'softLimit' => 0);
    return BEvent::Fire('shorten', array($doc, $length, &$options));
  }

  // $post - post path (e.g.: /cat/post.wiki). $text - wiki source.
  static function AddOrUpdate($post, &$text, $extraInfo = array()) {
    $func = self::Exists($post) ? 'Update' : 'Add';
    self::$func($post, $text, $extraInfo);
  }

  protected static function Add($post, &$text, $extraInfo = array()) {
    BEvent::Fire('normalize post', array(&$text, &$post, &$extraInfo));
    self::EnsureVacantPermalink($extraInfo['permalink'], $post);
    BEvent::Fire('post added', array($post, &$extraInfo));
  }

    static function EnsureVacantPermalink($permalink, $letPost = null) {
      if ($post = BIndex::SelectFrom('UrlTitles', $permalink)
          and $post !== $letPost) {
        $letPost = BConfig::ToUTF8('file name', $letPost);
        $post = BConfig::ToUTF8('file name', $post);
        throw new BException($letPost.' has the same permalink ('.$permalink.') as '.$post);
      }
    }

  protected static function Update($post, &$text, $extraInfo = array()) {
    if (!self::Exists($post)) {
      throw new BException('Post to update '.$post.' didn\'t exist.');
    }

    BEvent::Fire('normalize post', array(&$text, &$post, &$extraInfo));
    self::EnsureVacantPermalink($extraInfo['permalink'], $post);
    $oldInfo = self::Get($post);
    BEvent::Fire('post updated', array($post, &$extraInfo, &$oldInfo));
  }

  static function Delete($post) {
    if (!self::Exists($post)) {
      throw new BException('Post to delete '.$post.' didn\'t exist.');
    }

    $info = self::Get($post);
    BEvent::Fire('before deleting post', array($post, &$info));
    BEvent::Fire('post deleted', array($post, &$info));
  }

  static function Move($post, $newName) {
    if (!self::Exists($post)) {
      throw new BException('Post to move '.$post.' didn\'t exist.');
    }
    if (self::Exists($newName)) {
      throw new BException('New name "'.$newName.'" for post '.$post.' is already registered.');
    }

    $info = self::Get($post);
    BEvent::Fire('before moving post', array($post, &$newName, &$info));
    BEvent::Fire('post moved', array($post, $newName, &$info));
  }
}
