<?php
/*
  This action works with DateFmt class (http://proger.i-forge.net/DateFmt/cWq)).
  It accepts a format string, optinal timestamp and language and outputs them either
  as a table (%%DateFmt) or inline ({{DateFmt}}).

  {{DateFmt D__}}
  {{DateFmt d##my, 1294829352}}         - exact timestamp
  {{DateFmt d#my, @2010-01-12, ru}}     - exact timestamp but specified in YYYY-MM-DD format
  {{DateFmt h#m, +60}}                  - now + 60 seconds (always start relative time with + or -)
  {{DateFmt AGO[b], -3600 * 24, ru}}    - yesterday, output in Russian (default is English)
    - only digits, spaces and * + - . , ( ) are allowed (0,5 -> 0.5 - fractions).
  {{DateFmt AGO[d.h], -d - h*14 - 12}}
    - case-insensitive shortcuts for seconds: (d)ay = 60*60*24, (h)our = 60*60, (m)inute=60, (s)econd=1

    Note that this is NOT valid:  {{DateFmt -2d}} - multiplication is missing:  {{DateFmt -2*d}}

    Resulting formatted string will be formatted as %%Wacko.
    If format string didn't contain any datetime formatting and no timestamp was
    passed explicitly only wacko-formatted result will be output (with empty source/time columns).

  Formatter form can also be used (line with "---" outputs no date but formats it as %%Wacko):
  %%(DateFmt)
    (ru)  AGO[*]AT h## a.m.
          d#.mo##.y# H#:m#    = +3600*0,3
    ---
    (en)  AGO[b]IF-FAR[d#my]  = 1294829352
  %%

  CONFIGURATION - field added to UWikiSettings:

    * DateFmtClassPath - defaults to none, thus date.php must be placed into autoload/
*/

class Udatefmt_Root extends UWikiBaseElement {
  public $isFormatter = true;
  public $htmlTag = 'span';
  public $htmlClasses = array('datefmt');

  function Parse() {
    if ($format = $this->settings->format and !$this->raw) {
      $dates = array(array_keys( $format->current['params'] ));
    } else {
      $dates = $this->ParseRawParams($this->raw);
      $this->isBlock = true;
      $this->htmlTag = 'table';
    }

    if ($path = &$format->settings->DateFmtClassPath) {
      include_once $path;
    }

    if (!class_exists('DateFmt')) {
      $this->children[] = $this->NewElement('Udatefmt_NoDateFmtClass');
    } else {
      $nowStr = &$this->strings['{{datefmt: now'];
      self::IsEmptyStr($nowStr) and $nowStr = 'now';

      foreach ($dates as $date) {
        $result = $this->ChildFrom($date);

        if ($this->isBlock) {
          $row = $this->children[] = $this->NewElement('Udatefmt_Row');

          $src = $row->children[] = $this->NewElement('Udatefmt_Source');
          $time = $row->children[] = $this->NewElement('Udatefmt_Timestamp');

            if ($date[0] !== $result->raw) {
              if (!isset($date[1])) {
                $time->SetRaw($nowStr);
              } elseif ($date[0] !== $result->raw) {
                $time->SetRaw( isset($date[1]) ? $date[1] : $nowStr );
              }
            }

            $time->raw === $nowStr and $time->htmlClasses[] = 'now';

            if ($time->raw !== null) {
              $srcText = $date[0];
              $date[2] === 'en' or $srcText = "($date[2]) $srcText";
              $src->SetRaw($srcText);
            }

          $result->isBlock = true;
          $result->htmlTag = 'td';
          $row->children[] = $result;
        } else {
          $this->children[] = $result;
        }
      }
    }
  }

    function ChildFrom($date) {
      $child = $this->NewElement('Udatefmt_Date');

      $date = $this->NormalizeDate($date);

        if (is_array($date)) {
          try {
            $date = call_user_func_array(array('DateFmt', 'Format'), $date);
          } catch (Exception $e) {
            $date = $e->getMessage();
          }
        }

      if ($doc = UWikiDocument::TryParsing($date, $this->settings, 'wacko')) {
        $inline = $doc->root->FindInline();
        $child->children[] = $inline ? $inline : $doc->root;
      }

      $child->SetRaw($date);
      return $child;
    }

      function NormalizeDate($date) {
        $result = $date;

        if (self::IsEmptyStr($result[0])) {
            $msg = '{{datefmt: no pattern';
            $str = 'No pattern was passed to {{DateFmt}}. Parameters: {{DateFmt pattern[, time[, lang]]}}.';

            return isset( $this->strings[$msg] ) ? $this->strings[$msg] : $str;
          }

        self::IsEmptyStr($result[1]) and $result[1] = time();
        $result[1] = self::CalcTimestamp($result[1]);

          if (!is_numeric( $result[1] )) {
            $msg = '{{datefmt: wrong timestamp';
            $str = "Timestamp ($date[1]) is not a number or expression (only digits,".
                   ' spaces and * + - . , ( ) are allowed).';

            return isset( $this->strings[$msg] ) ? $this->strings[$msg] : $str;
          }

        isset($result[2]) or $result[2] = 'en';
        return $result;
      }

        static function CalcTimestamp($str) {
          if (is_numeric($str) and !strpbrk($str[0], '-+')) {
            $date = (int) $str;
          } elseif ($str[0] === '@') {
            $date = strtotime( substr($str, 1) );
          } else {
            $date = null;
          }

          if ($date === null) {
            $str = strtolower($str);
            // spaces are added to explicitly point out mistakes in expression like this:
            // "-2d" -> "-286400" which is not quite "-2*86400" = 172800.
            $str = strtr($str, array('w' => ' '.(3600 * 24 * 7).' ', 'd' => ' '.(3600 * 24).' ',
                                     'h' => ' 3600 ', 'm' => ' 60', 's' => ' 1 '));

            if (preg_match('~^[+-]?[0-9 *+\-.,()]+$~', $str)) {
              try {
                $str = strtr($str, ',', '.');
                $str = eval("return $str;");
                $date = is_numeric($str) ? time() + $str : null;
              } catch (Exception $e) {
              }
            }
          }

          return $date;
        }

    function ParseRawParams($raw) {
      $params = array();

        $raw = explode("\n", $raw);
        foreach ($raw as $line) {
          if (($line = trim($line)) === '') { continue; }

          $parts = explode('=', $line);
          if (isset($parts[1])) {
            $time = trim( array_pop($parts) );
          } else {
            $time = null;
          }

          $pattern = join('=', $parts);

          if (isset($pattern[0]) and $pattern[0] === '(' and
              preg_match('/^\(([a-zA-Z]{2,3})\)(?=.)/u', $pattern, $match)) {
            $lang = strtolower($match[1]);
            $pattern = substr($pattern, strlen($match[0]));
          } else {
            $lang = 'en';
          }

          $params[] = array($pattern, $time, $lang);
        }

      return $params;
    }
}

  class Udatefmt_NoDateFmtClass extends UWikiFormatErrorMessage {
    public $message = '{{datefmt: no class';
    public $defaultMessage = '{{DateFmt}} cannot run because DateFmt class is not loaded (see DateFmtClassPath setting).';
  }

class Udatefmt_Date extends UWikiBaseElement {
  public $htmlClasses = array('datefmt');
}

class Udatefmt_Row extends UWikiBaseElement {
  public $htmlTag = 'tr';
}

  class Udatefmt_Source extends Upre_Root {
    public $htmlTag = 'td';
    public $htmlClasses = array('source');
  }

    class Udatefmt_Timestamp extends Udatefmt_Source {
      public $htmlClasses = array('timestamp');
    }
