program WackoDoc;

{$APPTYPE CONSOLE}

uses
  Windows,
  SysUtils,
  Classes,
  StrUtils,
  ShellAPI;


  const
    Header              = '  ==  WackoDoc PHP wrapper  -  by Proger_XP  ==';
    TempDirBaseName     = 'wackodoc-temp';
    PackageResName      = 'Wrapper';
    PassThruExe         = 'php.exe';
    PassThruExeArgs     = '-f start.php --';  // dash stops CL parsing by PHP to pass the rest to the script.

  function PreparePassThruExeArgs(const Args: String): String;
  begin
    Result := PassThruExeArgs + ' --chdir="' + ExtractFilePath(ParamStr(0)) + '" ' + Args;
    // php-cli considers \" as escaping of ".
    Result := StringReplace(Result, '\"', '\\"', [rfReplaceAll]);
  end;


{$R *.res}

const
  PackFileNameLength  = 260;
  MaxFilesInPack      = 256;

type
  TPackFatItem = packed record
    Name: array[0..PackFileNameLength] of Char;
    Size: DWord;
  end;
  TPackFAT = array of TPackFatItem;
  TPackFileCount = Integer;

procedure ShowError(const Msg: String; Fmt: array of const);
begin
  WriteLn( Format(Msg, Fmt) );
end;

  function SplitPackage(Pack: TStream; Dest: String): Boolean;
  var
    I: TPackFileCount;
    Table: TPackFAT;
    F: TStream;
    FN: String;
  begin
    Pack.ReadBuffer(I, SizeOf(I));
    SetLength(Table, I);

    Pack.ReadBuffer(Table[0], I * SizeOf(TPackFatItem));

    Dest := IncludeTrailingPathDelimiter(Dest);
    for I := 0 to Length(Table) - 1 do
    begin
      FN := Dest + String(Table[I].Name);
      if not DirectoryExists( ExtractFilePath(FN) ) then
        ForceDirectories( ExpandFileName(ExtractFilePath(FN)) );

      F := TFileStream.Create(FN, fmCreate);
      try
        if Table[I].Size <> 0 then
          F.CopyFrom(Pack, Table[I].Size);
      finally
        F.Free;
      end;
    end;

    Result := Pack.Position >= Pack.Size;
  end;

procedure SplitPackageWarnUnlessAtEOF(Pack: TStream; const Dest: String);
begin
  if not SplitPackage(Pack, Dest) then
    ShowError('Warning: the package wasn''t fully extracted into %s.', [ ExcludeTrailingPathDelimiter(Dest) ]);
end;

function RemoveDirectory(Path: WideString): Boolean;
var
  SR: TWin32FindDataW;
  Handle: DWord;
begin
  Result := True;
  Path := IncludeTrailingPathDelimiter(Path);

  Handle := FindFirstFileW(PWideChar(Path + '*.*'), SR);
  if Handle <> INVALID_HANDLE_VALUE then
  begin
    repeat
      if (WideString(SR.cFileName) <> '.') and (WideString(SR.cFileName) <> '..') then
        if (SR.dwFileAttributes and FILE_ATTRIBUTE_DIRECTORY <> 0) then
          Result := RemoveDirectory(Path + SR.cFileName) and Result
          else
            Result := DeleteFileW(PWideChar(Path + SR.cFileName)) and Result
    until not FindNextFileW(Handle, SR);
    Windows.FindClose(Handle)
  end;

  Result := RemoveDirectoryW(PWideChar(Path)) and Result
end;

  function PassThruRunAndWaitIn(const AppPath: String): Boolean;
  var
    PhpArgs: String;
    StartInfo: TStartupInfo;
    ProcInfo: TProcessInformation;
    ExitCode: DWord;
  begin
    PhpArgs := GetCommandLine;
    if PhpArgs[1] = '"' then
      PhpArgs := Copy(PhpArgs, PosEx('"', PhpArgs, 2) + 1, $FFFFF)
      else if Pos(' ', PhpArgs) <> 0 then
        PhpArgs := Copy(PhpArgs, Pos(' ', PhpArgs) + 1, $FFFFF)
        else
          PhpArgs := '';

    PhpArgs := PreparePassThruExeArgs(PhpArgs);

    ZeroMemory(@ProcInfo, SizeOf(TProcessInformation));
    StartInfo.cb := SizeOf(TStartupInfo);
    ZeroMemory(@StartInfo, SizeOf(TStartupInfo));

    // ShellExecute won't do because it will detach the new process' console.
    Result := CreateProcess(PChar(AppPath + PassThruExe), PChar(PhpArgs), NIL, NIL,
                            False, 0, NIL, PChar(AppPath), StartInfo, ProcInfo);

    if Result and (ProcInfo.hProcess <> 0) then
      Result := (GetExitCodeProcess(ProcInfo.hProcess, ExitCode) or (ExitCode = STILL_ACTIVE))
                and (WaitForSingleObject(ProcInfo.hProcess, INFINITE) = WAIT_OBJECT_0);
  end;

procedure PassThru;
var
  TempDir: String;
  I: Integer;
  Res: TResourceStream;
begin
  SetLength(TempDir, MAX_PATH);
  SetLength(TempDir, GetTempPath(MAX_PATH, PChar(TempDir)));

  TempDir := ExpandFileName(TempDir);
  if TempDir <> '' then
    TempDir := IncludeTrailingPathDelimiter(TempDir);

  I := 1;
  while DirectoryExists(TempDir + TempDirBaseName + IntToStr(I)) and (I < 50) do
    Inc(I);
  TempDir := TempDir + TempDirBaseName + IntToStr(I) + PathDelim;

  ForceDirectories(TempDir);
  if not DirectoryExists(TempDir) then
    ShowError('Couldn''t create the temporary directory:'#10'%s', [TempDir])
    else
      try

        try
          Res := TResourceStream.Create(hInstance, PackageResName, RT_RCDATA);
        except
          ShowError('Couldn''t find %s inside this launcher.', [PackageResName]);
          Exit;
        end;

        try
          SplitPackageWarnUnlessAtEOF(Res, TempDir);
          if not PassThruRunAndWaitIn(TempDir) then
            ShowError('Error running %s.'#10'GetLastError = #%d: %s.',
                      [TempDir + PassThruExe, GetLastError, SysErrorMessage(GetLastError)]);
        finally
          Res.Free;
        end;

      finally
        RemoveDirectory(TempDir);
      end;
end;

  function FillPackTableAndDataFrom(Path: String; var Table: TPackFAT; Data: TStream;
    FileIndex: Integer = 0; BasePath: String = ''): Integer;
  const
    AllocTableBy = 100;
  var
    SR: TWin32FindData;
    Handle: DWord;
    F: TStream;
    FN: String;
  begin
    Path := IncludeTrailingPathDelimiter(Path);
    if BasePath <> '' then
      BasePath := IncludeTrailingPathDelimiter(BasePath);

    Handle := FindFirstFile(PChar(Path + '*.*'), SR);
    if Handle <> INVALID_HANDLE_VALUE then
      repeat
        FN := Path + SR.cFileName;

        if DirectoryExists(FN) then
        begin
          if (String(SR.cFileName) <> '.') and (String(SR.cFileName) <> '..') then
            FileIndex := FillPackTableAndDataFrom(FN, Table, Data, FileIndex, BasePath + SR.cFileName);
        end
          else
          begin
            F := TFileStream.Create(FN, fmOpenRead or fmShareDenyWrite);
            try
              Data.CopyFrom(F, 0);

              Table[FileIndex].Size := F.Size;
              FN := BasePath + SR.cFileName;
              Move(FN[1], Table[FileIndex].Name[0], Length(FN));

              Inc(FileIndex);
            finally
              F.Free;
            end;
          end;
      until not FindNextFile(Handle, SR);

    Windows.FindClose(Handle);
    Result := FileIndex;
  end;

procedure MakePackage(const FN: String; DirToAdd: String);
var
  Pack, Data: TStream;
  Table: TPackFAT;
  I: TPackFileCount;
begin
  Data := TMemoryStream.Create;
  Data.Size := 5 * 1024 * 1024;
  Data.Position := 0;

  try
    SetLength(Table, MaxFilesInPack);
    ZeroMemory(@Table[0], SizeOf(Table));
    SetLength( Table, FillPackTableAndDataFrom(DirToAdd, Table, Data) );

    Pack := TFileStream.Create(FN, fmCreate);
    try
      I := Length(Table);
      Pack.WriteBuffer(I, SizeOf(I));
      Pack.WriteBuffer(Table[0], I * SizeOf(TPackFatItem));

      Data.Size := Data.Position;
      Pack.CopyFrom(Data, 0);
    finally
      Pack.Free;
    end;
  finally
    Data.Free;
  end;
end;

begin
  WriteLn(Header);
  WriteLn;

  ChDir( ExtractFilePath(ParamStr(0)) );

  try
    if ParamStr(1) = 'p' then
      MakePackage(PackageResName + '.dat', ParamStr(2))
      else if ParamStr(1) = 's' then
        SplitPackageWarnUnlessAtEOF( TFileStream.Create(PackageResName + '.dat', fmOpenRead), ParamStr(2) )
        else
          PassThru;
  except
    on E: Exception do
      ShowError(E.Message, []);
  end;
end.
