<?php

class EBuildTask extends OtherTaskError { }

  class ENothingToBuild extends EBuildTask {
    public $src;

    function __construct($srcPath) { $this->src = $srcPath; }
    function __toString() { return "No files to build were found in {$this->src}"; }
  }

  class ErrorDuringFileBuilding extends EBuildTask {
    public $fileInfo, $exception;

    function __construct($fileInfo, $exception) {
      $this->fileInfo = $fileInfo;
      $this->exception = $exception;
    }

    function __toString() {
      return "An error has occurred during the processing of {$this->fileInfo[baseName]}".
             " (use -q or -y to ignore errors):\n{$this->exception}";
    }
  }

UWikiTask::$tasks[] = 'BuildTask';
class BuildTask extends BaseTask {
  public $srcPath, $srcInterface, $destPath, $destInterface;
  public $templater, $tplPath, $tplInterface;
  public $inExt, $outExt, $indexFile, $builtFileList;
  public $localSrcPath, $currentFile;  // is set in BuildAll while building a file.
  public $currentLang, $languages;     // passed by MultiLangBuildTask.
  public $tplPrepared = false;   // $tpl/_prepare.php is called before first page is formatted.

  static function CanHandle($cl) { return $cl->flags['b'] or $cl->flags['u']; }

  function CheckEnvironment() {
    $errors = array(
      '-b and -u can\'t be used together - deside which one you want.' =>
        $this->cl->flags['b'] and $this->cl->flags['u'],
      'Source directory not passed.' => IsEmptyStr($this->srcPath),
      'Source directory does not exist.' => $this->srcInterface and !$this->srcInterface->Exists(),
      'Destination not passed (no default value for non-local source).' => IsEmptyStr($this->destPath),
      'Destination already exists (see -q, -y and -r).' =>
        !$this->cl->flags['u']
        and !IsEmptyStr($this->destPath) and $this->destInterface and $this->destInterface->Exists()
        and !array_intersect( array('q', 'y', 'r'), array_keys($this->cl->flags) ),
      'Template directory not passed (no default value for non-local source).' => IsEmptyStr($this->tplPath),
      '"Remove destination" (-r) cannot be used with update mode (-u).' =>
        $this->cl->flags['u'] and $this->cl->flags['r']
    );

    EEnvironmentCheck::ThrowIfAny($errors);
  }

  function Run() {
    $this->UpdateProps();
    $this->InitInterfaces();
    $this->CheckEnvironment();

    $this->templater = new UWikiTemplater($this->tplInterface->LocalPath(), $this->cl);
    $this->templater->CheckEnvironment();

      if (IsEmptyStr($this->outExt)) {
        $this->cl->options['outext'] = $this->outExt = ExtOf($this->templater->tplFile);
      }
      if (!isset( $this->cl->options['link-ext'] )) {
        $this->cl->options['link-ext'] = $this->outExt;
      }

      if (strpos($this->outExt, '.htm') === 0 and !isset($this->cl->options['static'])) {
        $this->cl->options['static'] = true;
      }

    try {
      $delRem = $this->cl->options['delete-removed'];
      if (!isset($delRem) or !empty($delRem)) {
        $deleted = $this->srcInterface->RemovedFiles($this->destInterface);
        $this->destInterface->RemoveByList($deleted);
      }

      $this->CopyFiles();
      $this->BuildAll();
    } catch (Exception $ex) {
      $e = $ex;
    }

      $this->CleanUp();
      $this->localSrcPath = null;
      $this->currentFile = null;
      $this->templater = null;

    if (isset($e)) { throw $e; }
  }

    function UpdateProps() {
      $src = $this->cl->byIndex[0];
      $this->srcPath = rtrim(( IsEmptyStr(realpath($src)) ? $src : realpath($src) ), '\\/');

      $this->destPath = rtrim($this->cl->byIndex[1], '\\/');
      $this->tplPath = rtrim($this->cl->options['template'], '\\/');
      $this->tplUtilsFile = dirname(__FILE__).'/template-utils.php';

      if ($inExt = $this->cl->options['inext']) {
        $this->inExt = preg_quote( ltrim($inExt, '.'), '/' );
      }
      IsEmptyStr($this->inExt) and $this->inExt = 'wiki';

      $this->outExt = $this->cl->options['outext'];

      $this->indexFile = $this->options['index-file'];
      IsEmptyStr($this->indexFile) and $this->indexFile = 'index.'.$this->inExt;

      $this->languages = $this->cl->options['Languages'];
      $this->currentLang = $this->cl->options['Current-Language'];
    }

    function InitInterfaces() {
      $this->srcInterface = $this->InterfaceFor('source', 'src', $this->srcPath);
      $this->srcInterface->CheckEnvironment();

        if (get_class($this->srcInterface) === 'FileInterface') {
          IsEmptyStr($this->destPath) and $this->destPath = dirname($this->srcPath).'/HTML';
          IsEmptyStr($this->tplPath) and $this->tplPath = dirname($this->srcPath).'/Template';
        }

      $this->destInterface = $this->InterfaceFor('destination', 'dest', $this->destPath);
      $this->destInterface->CheckEnvironment();

      $this->tplInterface = $this->InterfaceFor('source', 'tpl', $this->tplPath);
      $this->tplInterface->CheckEnvironment();
    }

      function InterfaceFor($type, $optionName, $path) {
        static $defaults = array('source' => array('BaseSourceInterface', 'FileInterface'),
                                 'destination' => array('BaseDestinationInterface', 'FileDestInterface'));

        if ($forceClass = &$this->cl->options["$optionName-interface"] and $forceClass !== 'auto') {
          if (class_exists($forceClass) and is_subclass_of($forceClass, $defaults[$type][0])) {
            return new $forceClass($path, $this->cl);
          } else {
            throw new OtherTaskError("Interface $forceClass does not exist or is of invalid class.");
          }
        } else {
          foreach (UWikiTask::$interfaces[$type] as $class) {
            if (call_user_func(array($class, 'CanHandle'), $path)) {
              return new $class($path, $this->cl);
            }
          }

          return new $defaults[$type][1]($path, $this->cl);
        }
      }

    function CopyFiles() {
      if ($this->cl->flags['r']) {
        $in = $this->inExt;
        $srcFiles = FindFilesRecursiveIn($this->destPath, '/\.(wiki|wacko|'.$in.')$/');

          if (!empty($srcFiles) and !$this->cl->flags['y']) {
            $in = in_array($in, array('wiki', 'wacko')) ? '' : " or $in";
            throw new OtherTaskError('Destination directory '.$this->destPath.' contains'.
                                     ' some of wiki or wacko'.$in.' files - are you sure '.
                                     ' you want to remove it (maybe'.
                                     ' you\'ve confused src with dest path arguments)?'.
                                     ' Use -y flag to continue anyway.');
          }

        RmDirRecursive($this->destPath);
      }

      CopyDir($this->tplInterface->LocalPath(), $this->destInterface, array($this, 'CanCopy'));

      if ($this->cl->options['copy-all']) {
        CopyDir($this->srcInterface->LocalPath(), $this->destInterface,
                array($this, 'CopyAllMatcher'));
      }

      if ($files = $this->cl->options['copy']) {
        foreach ($files as $dest => $src) {
          if (is_dir($src)) {
            CopyDirInto($dest, $src, $this->destInterface);
          } else {
            $this->destInterface->CopyFile($src, $dest);
          }
        }
      }
    }

    function BuildAll() {
      $this->builtFileList = array();
      $localSrcPath = $this->localSrcPath = rtrim($this->srcInterface->LocalPath(), '\\/').'/';

      $srcFiles = $this->FindSourceFilesIn($localSrcPath);
        if (empty($srcFiles) and !$this->cl->flags['u']) { throw new ENothingToBuild($this->srcPath); }

      if (empty($srcFiles)) {
        $this->cl->ProgressMessage('Nothing to update here');
      } else {
        $this->cl->ProgressMessage('Starting to render '.count($srcFiles).' wiki files');
      }

      foreach ($srcFiles as $i => $baseName) {
        $destBaseName = $this->DestBaseNameFor($baseName);

        // + fields not listed below: parseTimeMSec, destSize, fullPageSize (size after inserting into template).
        $fileInfo = array('src' => $this->srcPath."/$baseName", 'srcSize' => filesize($localSrcPath.$baseName),
                          'dest' => $this->destPath."/$destBaseName", 'destBaseName' => $destBaseName,
                          'tempLocalPath' => $localSrcPath, 'tempLocalFile' => $localSrcPath.$baseName,
                          'baseName' => $baseName, 'error' => null,
                          'sourceURL' => $this->MakeUrlFor($baseName, 'source-url'),
                          'historyURL' => $this->MakeUrlFor($baseName, 'history-url'));
        $this->currentFile = $fileInfo;

          if ($this->cl->StopOnProgress(count($srcFiles), $i + 1, $fileInfo)) { break; }

        $parseTime = microtime(true);
          try {
            $page = $this->BuildFile($fileInfo);
          } catch (EUWikiLastPCRE $e) {
            if ($this->cl->flags['y']) {
              $fileInfo['error'] = $e;
            } else {
              throw new ErrorDuringFileBuilding($fileInfo, $e);
            }
          }
        $fileInfo['parseTimeMSec'] = (microtime(true) - $parseTime) * 1000;

        if (!$fileInfo['error']) {
          try {
            $this->destInterface->AddFile($destBaseName, $page);
          } catch (EDestInterfaceWrite $e) {
            if ($this->cl->flags['y']) {
              $fileInfo['error'] = $e;
            } else {
              throw new ErrorDuringFileBuilding($fileInfo, $e);
            }
          }
        }

        $this->builtFileList[ $fileInfo['src'] ] = $fileInfo;
      }
    }

      function DestBaseNameFor($baseName) {
        return preg_replace('~(([^\\/])\.\w{1,5})?$~', '\2'.$this->outExt, $baseName, 1);
      }

      function FindSourceFilesIn($path) {
        $srcFiles = $files = FindFilesRecursiveIn($path, array($this, 'IsSourceFile'));
        if ($this->cl->flags['u']) {
          $updatedFiles = $this->srcInterface->UpdatedFiles($this->destInterface);
          is_array($updatedFiles) and $files = array_intersect($files, $updatedFiles);
          if (!$this->cl->options['existing-only']) {
            $new = $this->srcInterface->NewFiles($this->destInterface);
            $new = array_intersect($srcFiles, $new);
            $files = array_unique(array_merge($new, $files));
          }
        }

        return array_values($files);
      }

      function BuildFile(&$fileInfo) {
          $file = $fileInfo['tempLocalFile'];

        $page = file_get_contents($file);
        if ($appendFile = $this->cl->options['prepend']) { $page = $page.file_get_contents($appendFile); }
        if ($appendFile = $this->cl->options['append']) { $page .= file_get_contents($appendFile); }

        if ($charset = $this->cl->options['in-charset']) { $page = iconv($charset, 'UTF-8//IGNORE', $page); }

          $doc = $this->cl->CreateUWikiDocument($page, $this->cl->options['markup'], 'html');
          $this->PrepareDocFor($fileInfo, $doc);

          $doc->Parse();

        $page = $doc->RenderIn('html');
        $fileInfo['destSize'] = strlen($page);

        $page = $this->templater->FillWith( compact('page', 'doc') + array('task' => $this) + $fileInfo );
        if ($charset = $this->cl->options['out-charset']) { $page = iconv('UTF-8', "$charset//IGNORE", $page); }

        $fileInfo['fullPageSize'] = strlen($page);
        return $page;
      }

        function PrepareDocFor(&$fileInfo, $doc) {
          $doc->settings->pager = new UWikiFilePager( $fileInfo['tempLocalPath'] );

            $doc->settings->AlterPathsBy('/'.$fileInfo['baseName']);
            if ($url = $options['base-url']) {
              $doc->settings->BaseURL(rtrim($url, '\\/').'/'.$doc->settings->BaseURL());
            }

          if (!$this->tplPrepared) {
            $this->tplPrepared = true;
            require_once $this->tplUtilsFile;
            $this->PrepareTplUsing($doc->settings);
          }
        }

          function PrepareTplUsing($settings) {
            // resolve relative path as we'll change current dir below;
            $_script = realpath( $this->templater->GetPath().'/_prepare.php' );
            if (is_file($_script)) {
              $_vars = array('task' => $this, 'strings' => $settings->strings, 'cl' => $this->cl,
                             'templater' => $this->templater, 'lang' => $this->currentLang);
              extract($_vars);
              require $_script;
            }
          }

  function CanCopy($baseName, $fullPath) {
    static $prefixes = array('/_', '\\_', '/.svn', '\\.svn');

    foreach ($prefixes as $prefix) {
      if (strpos($fullPath, $prefix) !== false) { return false; }
    }

    return true;
  }

  function CopyAllMatcher($baseName, $fullPath) {
    return ExtOf($baseName) !== '.'.$this->inExt and $this->CanCopy($baseName, $fullPath);
  }

  function IsSourceFile($baseName, $fullPath) {
    return ExtOf($baseName) === '.'.$this->inExt and $this->CanCopy($baseName, $fullPath);
  }

  function MakeUrlFor($baseName, $paramName) {
    $pattern = $this->cl->options[$paramName];
    if (!IsEmptyStr($pattern)) {
      $replace = array('{URL}' => urlencode($baseName), '{PATH}' => $baseName);
      $url = strtr($pattern, $replace);
      $url === $pattern and $url .= array_shift($replace);

      return $url;
    }
  }

  function CleanUp() {
    if ($this->templater) { $this->templater->CLeanUp(); }
    if ($this->srcInterface) { $this->srcInterface->CleanUp(); }
    if ($this->destInterface) { $this->destInterface->CleanUp(); }
    if ($this->tplInterface) { $this->tplInterface->CleanUp(); }
  }

  /* Utility method that might be used by template ($task->...) */
  function IndexFileNameOf($dir) {
    $dir = rtrim($dir, '\\/');
    $dir === '' or $dir .= '/';
    return $this->localSrcPath.$dir.$this->indexFile;
  }

  function TranslationExists($langCode, $file) {
    if ($path = $this->languages[ strtolower($langCode) ]) {
      // since all languages must use the same kind of src interface it's fine to use ours.
      return $this->srcInterface->FileExistsIn($path, $file);
    }
  }

  function ListTranslationsFor($file) {
    $langs = array();

      if ($this->languages) {
        foreach ($this->languages as $code => $path) {
          if ($this->TranslationExists($code, $file)) {
            $baseName = $this->currentFile['destBaseName'];
            $outExt = $this->outExt;
            isset( $outExt[0] ) and $baseName = substr( $baseName, 0, -1 * strlen($outExt) );

            $langs[$code] = "$code/$baseName".$this->cl->options['link-ext'];
          }
        }
      }

    return $langs;
  }

  function FindConfigFile($baseName) {
    $paths = empty( $this->cl->options['config-path'] ) ? array(UWikiRootPath.'/config')
                                                        : $this->cl->options['config-path'];

    while (($path = array_pop($paths)) !== null) {
      $file = rtrim($path, '\\/')."/$baseName";
      if (is_file($file)) { return $file; }
    }
  }

  function PagePathOf($page, $pager) {
    $pagePath = array();

      $dirs = explode('/', "/$page");
      if (array_pop($dirs) === $this->indexFile) {
        array_pop($dirs);
      }

      $dirURL = '';
      foreach ($dirs as $dir) {
        $dirURL .= "$dir/";
        $pagePath[] = array('url' => $dirURL, 'titleHTML' => $this->GetPageTitle($dirURL, $pager),
                            'indexFile' => $pager->IndexPageNameOf($dirURL, true));
      }

    return $pagePath;
  }

    function GetPageTitle($page, $pager, $html = true) {
      $title = $pager->GetTItleOf($page);
      if (is_object($title)) {
        $title = $title->ChildrenToHTML(false);
        $html or $title = strip_tags($title);
      }

      return IsEmptyStr($title) ? basename( rtrim($page, '\\/') ) : $title;
    }

  function MakeHtmlTree($current, $pager, $options) {
    $options += array('baseURL' => rtrim($this->cl->options['root-url'], '\\/').'/',
                      'imageURL' => '', 'rootIcon' => '',
                      'expanded' => array(), 'current' => array());
    if ($options['imageURL'] !== '') {
      $options['imageURL'] = rtrim($options['imageURL'], '\\/').'/';
    }

    $path = explode('/', $current);
    $file = array_shift($path);
    $options['current'][$file] = $options['expanded'][$file] = $path;

    $tree = $this->MakeHtmlTreePart($this->ListCluster('/', $pager), '/', $pager, $options);
    $rootCaption = $this->GetPageTitle('/', $pager);

    return <<<HTML
<fieldset>
  <legend class="root">
    <img src="$options[rootIcon]" alt="" />
    <a href="$options[baseURL]">$rootCaption</a>
  </legend>

  <div class="tree">$tree</div>
</fieldset>
HTML;
  }

    protected $pagerForSorting;
    protected function ListCluster($cluster, $pager) {
      $options = array('matchFuncs' => array(array($this, 'CanListPage')),
                       'ignoreByDefault' => true, 'sort' => array($this, 'SortCluster'));
      $this->pagerForSorting = $pager;
      return $pager->ListCluster($cluster, $options);
    }

      function CanListPage($file, $options) {
        return $this->CanCopy($file['name'], "/$file[name]");
      }

      function SortCluster($file_1, $file_2) {
        $pager = $this->pagerForSorting;
        return strcmp($this->GetPageTitle($file_1['name'], $pager, false),
                      $this->GetPageTitle($file_2['name'], $pager, false));
      }

    protected function MakeHtmlTreePart($files, $parentPath, $pager, $options) {
      $imageURL = $options['imageURL'];
      $baseURL = $options['baseURL'];

      $result = '<ul>';

      $i = 0;
      foreach ($files as $file => $info) {
        $isExpanded = isset($options['expanded'][$file]);
        $isCurrent = isset($options['current'][$file]);

        if (!$info['isDir']) {
          $pageURL = rtrim($baseURL, '/').$parentPath.$this->DestBaseNameFor($file);
          $expander = $children = '';
        } else {
          $pageURL = rtrim($baseURL, '/').$parentPath.$file;

          $img = $isExpanded ? 'collapse' : 'expand';
          $expander = "<a href='$pageURL' class='expander'><img src='$imageURL$img.png' /></a>";

          if ($isExpanded) {
            $subfiles = $this->ListCluster($parentPath.$file, $pager);

            $suboptions = $options;
            $suboptions['expanded'] = self::SublistPath($options['expanded']);
            $suboptions['current'] = self::SublistPath($options['current']);

            $children = $this->MakeHtmlTreePart($subfiles, $parentPath.$file.'/',
                                                $pager, $suboptions);
          } else {
            $children = '';
          }
        }

        $liClass = ((count($files) === ++$i) ? ' class="last"' : '');
        $icon = $info['isDir'] ? 'dir' : 'doc';
        $nameTag = $isCurrent ? 'strong' : 'span';
        $caption = $this->GetPageTitle($parentPath.$file, $pager);

        $result .= <<<HTML
<li$liClass>
  <span class="tracer">
    $expander
    <span class="name">
      <img src="$imageURL$icon.png" />
      <a href="$pageURL"><$nameTag>$caption</$nameTag></a>
    </span>
  </span>
  $children
</li>
HTML;
    }

    return $result.'</ul>';
  }

    // converts array( 'key' => array('path', 'sub', ...), ... ) to array( 'path' => array('sub', ...), ... )
    static function SublistPath($list) {
      $result = array();

      foreach ($list as $key => $value) {
        $key = array_shift($value);
        $key and $result[$key] = $value;
      }

      return $result;
    }
}
