<?php

class Permalink {
  static $strToUrl = array('@#$&*=+/\\' => '-',
                           ' .' => '_',
                           '[{' => '(',
                           ']}' => ')');
  static $strToUrlDelete = '?!%^\'"<>:;,|«»“”';

  static $tailSeparator = '/';
  static $tailAlphabet = 'XVZ3tGnr4MpAf0EIcO7eSq9TC5NwhuF1DKJ6B8RyUiHbdoPLksWljamvz2gxQY';

  static function GenerateFor(&$result, $doc, &$extraLinks, $post = null) {
    $result = trim(@$doc['permalink']);
    if ($result === '' and isset($doc['doc'])) {
      $result = trim( @$doc['doc']->meta['permalink'] );
    }
    if ($result === '') {
      $result = BPosts::FormatLinkFor($doc, BConfig::$permalinkFormat, $post);
    }

    if (self::IsTailEnabled($doc) and $date = $doc['dateWritten']) {
      $extraLinks[] = $result;

      $tail = self::TailAt($date, $post);
      $tail === null or $result .= self::$tailSeparator.$tail;
    }
  }

    static function IsTailEnabled($docMeta) {
      return ($docMeta and !empty($docMeta['tail'])) ? BConfig::ToBool($docMeta['tail'])
                                                     : BConfig::$autoTail;
    }

      static function IsTailEnabledFor($post) {
        return self::IsTailEnabled( BPosts::Get($post) );
      }

    static function TailAt($date, $post = null) {
      /* Structure of the tail (which is a number):
           366 for days in a year * 40 years span * 16 posts per day = 234240 values
         (actually it's 15 posts per day - last one is used to indicate "too many posts"
          and when it's encountered all posts for that date will be returned.)
         We can get 62 symbols in our URL-safe alphabet: 10 for digits and 26 * 2 for Latin
         chars. Thus:  root of index 3 (234240) = 61.6 => 62 symbols
         For explanations on how this math works see http://proger.i-forge.net/Short_MD5 */

      if ($post and BPosts::Exists($post)) {
        $posts = BPosts::By(array( array('dateday', BPosts::DateOf($post)) ));
        $seqNum = (int) array_search($post, $posts);
      } else {
        $seqNum = BPosts::By(array( array('dateday', $date) ));
        $seqNum = count($seqNum);
      }

      // why 'y', not 'Y"? To make start year 2000 instead of 1970.
      $tail = date('y', $date) * 366 * 16 + date('z', $date) * 16 + min($seqNum, 16 - 1);
      return DecToBase(self::$tailAlphabet, $tail);
    }

  static function ByTailOf($title, &$post) {
    $title = trim($title, '\\/');

      // [category/] post [/tail]   - post cannot contain /; tail is 1-3 chars long.
      if ($tail = self::TailFrom($title)) {
        $title = substr($title, 0, -1 * strlen($tail));
      }

      if (strpbrk($title, '\\/') !== false) {
        $category = dirname($title);
        $title = basename($title);
      } else {
        $category = null;
      }

    if ($title !== '') {
      $post = BIndex::SelectFrom('UrlTitles', mb_strtolower($title));
      if ($category !== null and !$post) {
        $post = BIndex::SelectFrom('UrlTitles', "$category/$title");
      }
    }

    if (!$post and $tail !== null) {
      if ($tail = self::ParseTail($tail)) {
        $posts = BPosts::By(array( array('dateday', $tail[0]) ));
        if (!empty($posts)) {
          $seqNum = $tail[1];
          // note: do not sort posts here - we specifically need the order in which they were added.
          $post = ($seqNum >= count($posts) or $seqNum >= 15) ? $posts : $posts[$seqNum];
        }
      }
    }

    if ($post) { return true; }
  }

    static function TailFrom($title) {
      $pos = strrpos($title, self::$tailSeparator);
      if ($pos + 3 >= strlen($title) - 1) {
        $tail = substr($title, $pos === false ? 0 : $pos + 1);
        return ltrim($tail, self::$tailAlphabet) === '' ? $tail : null;
      }
    }

    static function ParseTail($str) {
      if (($str = trim($str)) !== '') {
        try {
          $tail = (int) BaseToDec(self::$tailAlphabet, $str);
        } catch (Exception $e) {
          if (strpos($e->getMessage(), 'BaseToDec') !== false) {
            // malformed tail.
            return null;
          } else {
            throw $e;
          }
        }

        if ($tail) {
          $year = (int) ($tail / 366 / 16);
          strlen($year) === 1 and $year = "0$year";
          $day = (int) (($tail - $year * 366 * 16) / 16);
          $seqNum = (int) ($tail - $year * 366 * 16 - $day * 16);

          $timestamp = (int) strtotime("1 Jan 20$year +$day day");
          if ($timestamp > 0) {
            return array($timestamp, $seqNum);
          }
        }
      }
    }

  // a handler for 'format post link'.
  static function FormatLinkSimple(&$format, $info, $post = null) {
    $r = array();

      if (isset( $info['textTitle'] )) {
        $title = $info['textTitle'];
      } elseif ($info['doc'] instanceof UWikiDocument) {
        $title = $info['doc']->HtmlTitle();
      } else {
        $title = '';
      }

        $r['$rawTitle'] = $title;
        $r['$title'] = self::GenerateFrom($title);

      $r['$category'] = join('/', $info['category']);
      $r['$category'] === '' or $r['$category'] .= '/';

        foreach (array('tags') as $field) {
          $r['$'.$field] = strtr8(join('_', $info[$field]), ' ', '-');
        }

      $r['$timestamp'] = $info['postTime'];
      if ($r['$timestamp']) {
        static $dateFormats = array('dateDay' => 'd', 'dateMonth' => 'm', 'dateYear2' => 'y',
                                    'dateYear4' => 'Y', 'timeSec' => 's', 'timeMin' => 'i',
                                    'timeHour24' => 'H', 'timeHour12' => 'h', 'timeHourAM' => 'A');
        foreach ($dateFormats as $field => $fmt) {
          $r['$'.$field] = date($fmt, $info['postTime']);
        }
      }

      if (empty( $info['source'] )) {
        $r['$Chars1000'] = 0;
      } else {
        $r['$Chars1000'] = round( mb_strlen($info['source']) / 1000 );
      }

      if ($post === null) {
        $r += array('$path' => '', '$file' => '', '$ext' => '');
      } else {
        $r['$ext'] = ltrim(ExtOf($post), '.');
        $r['$file'] = basename($post, '.'.$r['$ext']);

        $path = ltrim(strtr8($post, '\\', '/'), '/');
        $r['$path'] = mb_strpos($path, '/') ? dirname($path).'/' : '';
      }

    $format = strtr($format, $r);
  }

    static function GenerateFrom($str) {
      $delete = '~^\s+|\s+$|[\x00-\x1F'.preg_quote(self::$strToUrlDelete, '~').']+~u';
      $conv = array($delete => '');

      foreach (self::$strToUrl as $from => $to) {
        $conv['~['.preg_quote($from, '~').']+~u'] = $to;
      }

      $str = preg_replace(array_keys($conv), array_values($conv), $str);

      $options = array('max' => BConfig::$maxPermalinkLength, 'soft' => 10, 'cut' => '');
      return ShortenInlineText($str, $options);
    }
}
